/*
 *  GVDR library for reading GVDR data files
 *  Copyright (C) 1994 Michael J. Maurer
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Michael Maurer <maurer@nova.stanford.edu>
 *  Durand Bldg - Room 232
 *  Stanford, CA 94305-4055
 *  (415) 723-1024
 */
static char rcsid[]="$Id: gcvt.c,v 1.1 1993/08/02 00:57:29 maurer Exp $";
/******************************************************************************
  gcvt.c

     Function: Generic data type converter.  Converts between local machine
representations and foreign machine representations.

This file is part of the STARLab Magellan Altimeter Data Processing Software.
Michael Maurer, May 1993.
******************************************************************************/
/* $Log: gcvt.c,v $
 * Revision 1.1  1993/08/02  00:57:29  maurer
 * Modified to use ivconvert.c.
 * Removed map.h and util.h for public distribution.
 *
 * Revision 1.0  1993/06/12  00:36:36  maurer
 * Initial revision
 * */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "libmisc.h"
#include "gcvt.h"

#define GCVT_C
#include "gcvt.p"

static int gcvt_dir	= GCVT_WRITE;
static int gfmt_float	= GFMT_IEEE;
static int gfmt_byte	= GFMT_BIGEND;

void
gcvt_setmode(int dir,
	     int fmt_float,
	     int fmt_byte)
{
   gcvt_dir=dir;
   gfmt_float=fmt_float;
   gfmt_byte=fmt_byte;
}

/******************************************************************************
  gcvt_s

If gcvt_dir==GCVT_WRITE, converts n short ints from local (native) machine
format to the format specified in global variable gfmt_byte.  If
gcvt_dir==GCVT_READ, the direction of conversion is reversed.  src points
to the first source element, and dst to the first destination element.
src and dst may have the same value (and the conversion will be
performed in place).  However, src and dst may not overlap in more
general ways.
******************************************************************************/

void
gcvt_s(src,dst,n)
void *src,*dst;
int n;
{
   switch (gcvt_dir) {
    case GCVT_WRITE:
      switch (gfmt_byte) {
       case GFMT_BIGEND:
	 s_local2ieee(src,dst,n);
	 break;
       case GFMT_LITEND:
	 s_local2mipsel(src,dst,n);
	 break;
       default:
	 error(-1,0,"[gcvt_s] unknown byte format %d",gfmt_byte);
      }
      break;
    case GCVT_READ:
      switch (gfmt_byte) {
       case GFMT_BIGEND:
	 s_ieee2local(src,dst,n);
	 break;
       case GFMT_LITEND:
	 s_mipsel2local(src,dst,n);
	 break;
       default:
	 error(-1,0,"[gcvt_s] unknown byte format %d",gfmt_byte);
      }
      break;
    default:
      error(-1,0,"[gcvt_s] unknown conversion direction %d",gcvt_dir);
   }
}

/******************************************************************************
  gcvt_l

If gcvt_dir==GCVT_WRITE, converts n long ints from local (native) machine
format to the format specified in global variable gfmt_byte.  If
gcvt_dir==GCVT_READ, the direction of conversion is reversed.  src points
to the first source element, and dst to the first destination element.
src and dst may have the same value (and the conversion will be
performed in place).  However, src and dst may not overlap in more
general ways.
******************************************************************************/

void
gcvt_l(src,dst,n)
void *src,*dst;
int n;
{
   switch (gcvt_dir) {
    case GCVT_WRITE:
      switch (gfmt_byte) {
       case GFMT_BIGEND:
	 l_local2ieee(src,dst,n);
	 break;
       case GFMT_LITEND:
	 l_local2mipsel(src,dst,n);
	 break;
       default:
	 error(-1,0,"[gcvt_l] unknown byte format %d",gfmt_byte);
      }
      break;
    case GCVT_READ:
      switch (gfmt_byte) {
       case GFMT_BIGEND:
	 l_ieee2local(src,dst,n);
	 break;
       case GFMT_LITEND:
	 l_mipsel2local(src,dst,n);
	 break;
       default:
	 error(-1,0,"[gcvt_l] unknown byte format %d",gfmt_byte);
      }
      break;
    default:
      error(-1,0,"[gcvt_l] unknown conversion direction %d",gcvt_dir);
   }
}

/******************************************************************************
  gcvt_f

If gcvt_dir==GCVT_WRITE, converts n floats from local (native) machine
format to the format specified in global variables gfmt_byte and
gfmt_float.  If gcvt_dir==GCVT_READ, the direction of conversion is
reversed.  src points to the first source element, and dst to the first
destination element.  src and dst may have the same value (and the
conversion will be performed in place).  However, src and dst may not
overlap in more general ways.
******************************************************************************/

void
gcvt_f(src,dst,n)
void *src,*dst;
int n;
{
   switch (gcvt_dir) {
    case GCVT_WRITE:
      switch (gfmt_byte) {
       case GFMT_BIGEND:
	 switch (gfmt_float) {
	  case GFMT_IEEE:
	    f_local2ieee(src,dst,n);
	    break;
	  case GFMT_VAX:
	    f_local2vax(src,dst,n);
	    swapb_4(dst,dst,n);
	    break;
	  default:
	    error(-1,0,"[gcvt_f] unknown float format %d",gfmt_float);
	 }
	 break;
       case GFMT_LITEND:
	 switch (gfmt_float) {
	  case GFMT_IEEE:
	    f_local2ieee(src,dst,n);
	    swapb_4(dst,dst,n);
	    break;
	  case GFMT_VAX:
	    f_local2vax(src,dst,n);
	    break;
	  default:
	    error(-1,0,"[gcvt_f] unknown float format %d",gfmt_float);
	 }
	 break;
       default:
	 error(-1,0,"[gcvt_f] unknown byte format %d",gfmt_byte);
      }
      break;
    case GCVT_READ:
      switch (gfmt_byte) {
       case GFMT_BIGEND:
	 switch (gfmt_float) {
	  case GFMT_IEEE:
	    f_ieee2local(src,dst,n);
	    break;
	  case GFMT_VAX:
	    swapb_4(src,dst,n);
	    f_vax2local(dst,dst,n);
	    break;
	  default:
	    error(-1,0,"[gcvt_f] unknown float format %d",gfmt_float);
	 }
	 break;
       case GFMT_LITEND:
	 switch (gfmt_float) {
	  case GFMT_IEEE:
	    swapb_4(src,dst,n);
	    f_ieee2local(dst,dst,n);
	    break;
	  case GFMT_VAX:
	    f_vax2local(src,dst,n);
	    break;
	  default:
	    error(-1,0,"[gcvt_f] unknown float format %d",gfmt_float);
	 }
	 break;
       default:
	 error(-1,0,"[gcvt_f] unknown byte format %d",gfmt_byte);
      }
      break;
    default:
      error(-1,0,"[gcvt_f] unknown conversion direction %d",gcvt_dir);
   }
}

/******************************************************************************
  gcvt_d

If gcvt_dir==GCVT_WRITE, converts n doubles from local (native) machine
format to the format specified in global variables gfmt_byte and
gfmt_float.  If gcvt_dir==GCVT_READ, the direction of conversion is
reversed.  src points to the first source element, and dst to the first
destination element.  src and dst may have the same value (and the
conversion will be performed in place).  However, src and dst may not
overlap in more general ways.
******************************************************************************/

void
gcvt_d(src,dst,n)
void *src,*dst;
int n;
{
   switch (gcvt_dir) {
    case GCVT_WRITE:
      switch (gfmt_byte) {
       case GFMT_BIGEND:
	 switch (gfmt_float) {
	  case GFMT_IEEE:
	    d_local2ieee(src,dst,n);
	    break;
	  case GFMT_VAX:
	    d_local2vax(src,dst,n);
	    swapb_8(dst,dst,n);
	    break;
	  default:
	    error(-1,0,"[gcvt_d] unknown float format %d",gfmt_float);
	 }
	 break;
       case GFMT_LITEND:
	 switch (gfmt_float) {
	  case GFMT_IEEE:
	    d_local2ieee(src,dst,n);
	    swapb_8(dst,dst,n);
	    break;
	  case GFMT_VAX:
	    d_local2vax(src,dst,n);
	    break;
	  default:
	    error(-1,0,"[gcvt_d] unknown float format %d",gfmt_float);
	 }
	 break;
       default:
	 error(-1,0,"[gcvt_d] unknown byte format %d",gfmt_byte);
      }
      break;
    case GCVT_READ:
      switch (gfmt_byte) {
       case GFMT_BIGEND:
	 switch (gfmt_float) {
	  case GFMT_IEEE:
	    d_ieee2local(src,dst,n);
	    break;
	  case GFMT_VAX:
	    swapb_8(src,dst,n);
	    d_vax2local(dst,dst,n);
	    break;
	  default:
	    error(-1,0,"[gcvt_d] unknown float format %d",gfmt_float);
	 }
	 break;
       case GFMT_LITEND:
	 switch (gfmt_float) {
	  case GFMT_IEEE:
	    swapb_8(src,dst,n);
	    d_ieee2local(dst,dst,n);
	    break;
	  case GFMT_VAX:
	    d_vax2local(src,dst,n);
	    break;
	  default:
	    error(-1,0,"[gcvt_d] unknown float format %d",gfmt_float);
	 }
	 break;
       default:
	 error(-1,0,"[gcvt_d] unknown byte format %d",gfmt_byte);
      }
      break;
    default:
      error(-1,0,"[gcvt_s] unknown conversion direction %d",gcvt_dir);
   }
}
