/*
 *  GVDR library for reading GVDR data files
 *  Copyright (C) 1994 Michael J. Maurer
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Michael Maurer <maurer@nova.stanford.edu>
 *  Durand Bldg - Room 232
 *  Stanford, CA 94305-4055
 *  (415) 723-1024
 */
static char rcsid[]="$Id: gv_pack.c,v 1.9 1994/03/04 22:06:10 maurer Exp $";
/******************************************************************************
  gv_pack.c

     Function:  Compress and expand GVDR data objects.

This file is part of the STARLab Magellan Altimeter Data Processing Software.
Michael Maurer, May 1993.
******************************************************************************/
/* $Log: gv_pack.c,v $
 * Revision 1.9  1994/03/04  22:06:10  maurer
 * Split into generic module gpack.c, now uses data structures.
 *
 * Revision 1.8  1994/02/02  02:13:34  maurer
 * Changed some packing to/from linear/logarithmic.
 *
 * Revision 1.7  1994/01/23  23:01:25  maurer
 * Cosmetic changes.
 *
 * Revision 1.6  1994/01/06  00:18:47  maurer
 * Added conversion for ga_azim.
 *
 * Revision 1.5  1993/09/10  19:39:11  maurer
 * Added gscaling_*().
 *
 * Revision 1.4  1993/08/25  21:20:11  maurer
 * Moved UCHAR_* and USHRT_* from gv_pack.c to gv_pack.h.
 * Added ADF support to cell_rlen().
 * Added ADF packing/unpacking functions.
 *
 * Revision 1.3  1993/08/02  01:00:56  maurer
 * Changed uchar -> unsigned char, etc.
 * Removed map.h and util.h for public distribution.
 *
 * Revision 1.2  1993/07/30  02:02:12  maurer
 * Added packing for new indices.
 *
 * Revision 1.1  1993/07/03  23:47:41  maurer
 * Updated to use new GVDR record formats, especially the change from
 * sfloat to 1-byte packed quantities.
 *
 * Revision 1.0  1993/06/12  00:38:57  maurer
 * Initial revision
 * */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "libmisc.h"
#include "gvdr.h"
#include "gpack.h"
#include "gv_pack.h"

#define GVDR_PACK_C
#include "gv_pack.p"
#include "gpack.p"


/******************************************************************************
  GVDR pack/unpack functions

******************************************************************************/

int
cell_rlen(gvdrcell_t *C)
{
   int i,reclen;

   reclen = sizeof(gvdrcell_rec);
   reclen += C->gv_Nxif*sizeof(gvdrxif_rec);
   reclen += C->gv_Nedf*sizeof(gvdredf_rec);
   reclen += C->gv_Nadf*sizeof(gvdradf_rec);
   for (i=0; i<C->gv_Nanf; i++) {
      reclen += sizeof(gvdranf_rec);
      reclen += 2 * C->gv_anf[i].ga_Nsig0 * sizeof(unsigned char);
      reclen += C->gv_anf[i].ga_Nfit * sizeof(gvdrfit_rec);
   }
   return reclen;
}

void
gpack_cell(gvdrcell_t *C,
	   gvdrcell_rec *Cr)
{
   if (C->gv_Nxif>0xFF || C->gv_Nedf>0xFF || C->gv_Nanf>0xFF)
      error(-1,0,"[gpack_cell] uchar overflow");
   Cr->gvr_reclen = cell_rlen(C);
   Cr->gvr_Nxif = C->gv_Nxif;
   Cr->gvr_Nedf = C->gv_Nedf;
   Cr->gvr_Nadf = C->gv_Nadf;
   Cr->gvr_Nanf = C->gv_Nanf;
}

void
gunpack_cell(gvdrcell_rec *Cr,
	     gvdrcell_t *C)
{
   C->gv_reclen = Cr->gvr_reclen;
   C->gv_Nxif = Cr->gvr_Nxif;
   C->gv_Nedf = Cr->gvr_Nedf;
   C->gv_Nadf = Cr->gvr_Nadf;
   C->gv_Nanf = Cr->gvr_Nanf;
}

void
gpack_xif(gvdrxif_t *g,
	  gvdrxif_rec *gr)
{
   gr->gxr_Nlooks	= g->gx_Nlooks;
   gr->gxr_azim		= gpack(g->gx_azim,rg_azim);
   gr->gxr_inc		= gpack(g->gx_inc,rg_inc);
   gr->gxr_poln		= gpack(g->gx_poln,rg_poln);
   gr->gxr_hlo		= g->gx_hlo;
   gr->gxr_hmed		= g->gx_hmed;
   gr->gxr_hhi		= g->gx_hhi;
   gr->gxr_hpk		= g->gx_hpk;
   gr->gxr_c0		= gpack(g->gx_c0,rg_c0);
   gr->gxr_c1		= gpack(g->gx_c1,rg_c1);
   gr->gxr_c2		= gpack(g->gx_c2,rg_c2);
}

void
gunpack_xif(gvdrxif_rec *gr,
	    gvdrxif_t *g)
{
   g->gx_Nlooks		= gr->gxr_Nlooks;
   g->gx_azim		= gunpack(gr->gxr_azim,rg_azim);
   g->gx_inc		= gunpack(gr->gxr_inc,rg_inc);
   g->gx_poln		= gunpack(gr->gxr_poln,rg_poln);
   g->gx_hlo		= gr->gxr_hlo;
   g->gx_hmed		= gr->gxr_hmed;
   g->gx_hhi		= gr->gxr_hhi;
   g->gx_hpk		= gr->gxr_hpk;
   g->gx_c0		= gunpack(gr->gxr_c0,rg_c0);
   g->gx_c1		= gunpack(gr->gxr_c1,rg_c1);
   g->gx_c2		= gunpack(gr->gxr_c2,rg_c2);
}

void
gpack_edf(gvdredf_t *g,
	  gvdredf_rec *gr)
{
   gr->ger_Nlooks	= g->ge_Nlooks;
   gr->ger_azim		= gpack(g->ge_azim,rg_azim);
   gr->ger_inc		= gpack(g->ge_inc,rg_inc);
   gr->ger_poln		= gpack(g->ge_poln,rg_poln);
   gr->ger_emiss	= gpack(g->ge_emiss,rg_emiss);
   gr->ger_emissv	= gpack(g->ge_emissv,rg_emissv);
}

void
gunpack_edf(gvdredf_rec *gr,
	    gvdredf_t *g)
{
   g->ge_Nlooks		= gr->ger_Nlooks;
   g->ge_azim		= gunpack(gr->ger_azim,rg_azim);
   g->ge_inc		= gunpack(gr->ger_inc,rg_inc);
   g->ge_poln		= gunpack(gr->ger_poln,rg_poln);
   g->ge_emiss		= gunpack(gr->ger_emiss,rg_emiss);
   g->ge_emissv		= gunpack(gr->ger_emissv,rg_emissv);
}

void
gpack_adf(gvdradf_t *g,
	  gvdradf_rec *gr)
{
   gr->gdr_Nlooks	= g->gd_Nlooks;
   gr->gdr_radius	= gpack(g->gd_radius,rg_radius);
   gr->gdr_radiusv	= gpack(g->gd_radiusv,rg_radiusv);
   gr->gdr_slope	= gpack(g->gd_slope,rg_slope);
   gr->gdr_slopev	= gpack(g->gd_slopev,rg_slopev);
   gr->gdr_rho		= gpack(g->gd_rho,rg_rho);
   gr->gdr_rhov		= gpack(g->gd_rhov,rg_rhov);
}

void
gunpack_adf(gvdradf_rec *gr,
	    gvdradf_t *g)
{
   g->gd_Nlooks		= gr->gdr_Nlooks;
   g->gd_radius		= gunpack(gr->gdr_radius,rg_radius);
   g->gd_radiusv	= gunpack(gr->gdr_radiusv,rg_radiusv);
   g->gd_slope		= gunpack(gr->gdr_slope,rg_slope);
   g->gd_slopev		= gunpack(gr->gdr_slopev,rg_slopev);
   g->gd_rho		= gunpack(gr->gdr_rho,rg_rho);
   g->gd_rhov		= gunpack(gr->gdr_rhov,rg_rhov);
}

void
gpack_anf(gvdranf_t *g,
	  gvdranf_rec *gr,
	  unsigned char *sig0,
	  unsigned char *Vsig0,
	  gvdrfit_rec *gf)
{
   int i;

   gr->gar_reclen	= g->ga_Nsig0*rg_sig0.nb;
   gr->gar_reclen	+= g->ga_Nsig0*rg_Vsig0.nb;
   gr->gar_reclen	+= g->ga_Nfit*sizeof(gvdrfit_rec);
   gr->gar_Nlooks	= g->ga_Nlooks;
   gr->gar_Nsig0	= g->ga_Nsig0;
   gr->gar_Nfit		= g->ga_Nfit;
   gr->gar_dcent	= gpack(g->ga_dcent,rg_dcent);
   gr->gar_azim		= gpack(g->ga_azim,rg_gazim);
#ifndef NEWANF
   gr->gar_pad		= 0;
#endif
   if (sig0 && Vsig0)
      for (i=0; i<g->ga_Nsig0; i++) {
	 sig0[i]	= gpack(g->ga_sig0[i],rg_sig0);
	 Vsig0[i]	= gpack(g->ga_Vsig0[i],rg_Vsig0);
      }
   if (gf)
      for (i=0; i<g->ga_Nfit; i++)
	 gpack_fit(&g->ga_fit[i],&gf[i]);
}

void
gunpack_anf(gvdranf_rec *gr,
	    unsigned char *sig0,
	    unsigned char *Vsig0,
	    gvdrfit_rec *gf,
	    gvdranf_t *g)
{
   int i;

   g->ga_Nlooks		= gr->gar_Nlooks;
   g->ga_Nsig0		= gr->gar_Nsig0;
   g->ga_Nfit		= gr->gar_Nfit;
   g->ga_dcent		= gunpack(gr->gar_dcent,rg_dcent);
   g->ga_azim		= gunpack(gr->gar_azim,rg_gazim);
   if (sig0 && Vsig0)
      for (i=0; i<g->ga_Nsig0; i++) {
	 g->ga_sig0[i]	= gunpack(sig0[i],rg_sig0);
	 g->ga_Vsig0[i]	= gunpack(Vsig0[i],rg_Vsig0);
      }
   if (gf)
      for (i=0; i<g->ga_Nfit; i++)
	 gunpack_fit(&gf[i],&g->ga_fit[i]);
}

void
gpack_fit(gvdrfit_t *g,
	  gvdrfit_rec *gr)
{
   gr->gfr_slaw		= g->gf_slaw;
   gr->gfr_flags	= g->gf_flags;
   gr->gfr_p1		= gpack(g->gf_p1,*rg_p1[g->gf_slaw]);
   gr->gfr_p1v		= gpack(g->gf_p1v,*rg_p1v[g->gf_slaw]);
   gr->gfr_p2		= gpack(g->gf_p2,rg_p2);
   gr->gfr_p2v		= gpack(g->gf_p2v,rg_p2v);
   gr->gfr_rms		= gpack(g->gf_rms,rg_rms);
   gr->gfr_rmsv		= gpack(g->gf_rmsv,rg_rmsv);
   gr->gfr_resid	= gpack(g->gf_resid,rg_resid);
   gr->gfr_pad		= 0;
}

void
gunpack_fit(gvdrfit_rec *gr,
	    gvdrfit_t *g)
{
   g->gf_slaw		= gr->gfr_slaw;
   g->gf_flags		= gr->gfr_flags;
   g->gf_p1		= gunpack(gr->gfr_p1,*rg_p1[gr->gfr_slaw]);
   g->gf_p1v		= gunpack(gr->gfr_p1v,*rg_p1v[gr->gfr_slaw]);
   g->gf_p2		= gunpack(gr->gfr_p2,rg_p2);
   g->gf_p2v		= gunpack(gr->gfr_p2v,rg_p2v);
   g->gf_rms		= gunpack(gr->gfr_rms,rg_rms);
   g->gf_rmsv		= gunpack(gr->gfr_rmsv,rg_rmsv);
   g->gf_resid		= gunpack(gr->gfr_resid,rg_resid);
}

void
gpack_indexh(gvdr_index_t *I,
	     gvdr_index_rec *Ir)
{
   Ir->lo = I->lo;
   Ir->hi = I->hi;
   Ir->nb = I->nb;
   Ir->mult = I->mult;
}

void
gunpack_indexh(gvdr_index_rec *Ir,
	       gvdr_index_t *I)
{
   I->lo = Ir->lo;
   I->hi = Ir->hi;
   I->nb = Ir->nb;
   I->mult = Ir->mult;
}

