/*
 *  GVDR library for reading GVDR data files
 *  Copyright (C) 1994 Michael J. Maurer
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Michael Maurer <maurer@nova.stanford.edu>
 *  Durand Bldg - Room 232
 *  Stanford, CA 94305-4055
 *  (415) 723-1024
 */
static char rcsid[]="$Id: gv_sfdu.c,v 1.6 1994/04/29 02:01:26 maurer Exp $";
/******************************************************************************
  gv_sfdu.c

     Function:  Handle SFDU reading/writing for GVDR files.

This file is part of the STARLab Magellan Altimeter Data Processing Software.
Michael Maurer, May 1993.
******************************************************************************/
/* $Log: gv_sfdu.c,v $
 * Revision 1.6  1994/04/29  02:01:26  maurer
 * Fixed sscanf format string.
 *
 * Revision 1.5  1993/11/24  18:32:37  maurer
 * Now correctly fills entire gvdr_sfducat_t structure when reading
 * in an SFDU keyword=value label.
 *
 * Revision 1.4  1993/09/10  19:39:24  maurer
 * Changed gsfdu_parsehdr() syntax.
 *
 * Revision 1.3  1993/08/25  21:21:50  maurer
 * Now gsfdu_parsehdr() reads the DATA_FORMAT_TYPE and the
 * INPUT_SEQUENCE_NUMBERS keywords and parses their values.
 *
 * Revision 1.2  1993/08/02  01:01:34  maurer
 * Split into reader/writer modules.
 *
 * Revision 1.1  1993/07/30  02:02:21  maurer
 * Removed SFDU from GVDR cell.
 *
 * Revision 1.0  1993/06/12  00:39:01  maurer
 * Initial revision
 * */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "libmisc.h"
#include "sfdug.h"
#include "gv_sfdu.h"
#include "gvdr.h"
#include "gcvt.h"

#define GVDR_SFDU_C
#include "gv_sfdu.p"
#include "gvdr.p"
#include "sfdug.p"
#include "gcvt.p"

/******************************************************************************
  gsfdu_rhdr

Reads all four SFDU header records, and the catalogs and binary record.
Returns nonzero on failure.
******************************************************************************/

int
gsfdu_rhdr(FILE *fp,
	   gvdrhdr_t *h)
{
   fpos_t fpos0,fpos1;
   unsigned long reclen;

   if (fgetpos(fp,&fpos0))
      return 1;
   if (gsfdu_rhdr1(fp,&h->sf1))
      return 2;
   if (gsfdu_rhdr2(fp,&h->sf2,&h->kv2))
      return 3;
   if (gsfdu_rhdr3(fp,&h->sf3,&h->hdr3))
      return 4;
   if (gsfdu_rhdr4(fp,&h->sf4,&h->kv4))
      return 5;
   reclen=h->hdr3.gh_rec_tile;
   gcvt_l(&reclen,&reclen,1);
   if (gvdr_rpad(fp,reclen,GVDR_PAD,&fpos1))
      return 6;
   if (sf_len(&h->sf1) != fpos1-fpos0-sizeof(sfdu_t))
      return 7;
   return 0;
}

/******************************************************************************
  gsfdu_rhdr1

Reads SFDU header #1 into sf.  Returns nonzero on failure.
******************************************************************************/

static int
gsfdu_rhdr1(FILE *fp,
	    sfdu_t *sf)
{
   if (fread(sf,sizeof(sfdu_t),1,fp)!=1) {
      error(0,errno,"[gsfdu_rhdr1] fread failed");
      return 1;
   }
   return 0;
}

/******************************************************************************
  gsfdu_rhdr2

Reads SFDU header #1 into sf.  Fills kv with keyword=value catalog
entries.  Returns nonzero on failure.
******************************************************************************/

static int
gsfdu_rhdr2(FILE *fp,
	    sfdu_t *sf,
	    keyval_t **kv)
{
   int len;
   char *buf=NULL;

   if ((len=sf_read(fp,sf,&buf))<=0)
      return 1;
   *kv=NULL;
   if (kv_parse(buf,kv)<0)
      return 2;
   free(buf);
   return 0;
}

/******************************************************************************
  gsfdu_rhdr3

Reads SFDU header #3 into sf.  Reads binary record into h.  Returns
nonzero on failure.
******************************************************************************/

static int
gsfdu_rhdr3(FILE *fp,
	    sfdu_t *sf,
	    gvdr_sfduhdr_t *h)
{
   if (sf_read(fp,sf,(char **)&h) != sizeof(gvdr_sfduhdr_t))
      return 1;
   return 0;
}

/******************************************************************************
  gsfdu_rhdr4

Reads SFDU header #4 into sf.  Reads start marker catalog into kv.a
Returns nonzero on failure.
******************************************************************************/

static int
gsfdu_rhdr4(FILE *fp,
	    sfdu_t *sf,
	    keyval_t **kv)
{
   return gsfdu_rhdr2(fp,sf,kv);
}

/******************************************************************************
  gsfdu_parsehdr

Parses the keyword=value catalogs in kv into data structure gc.  Returns
nonzero on failure.
******************************************************************************/

int
gsfdu_parsehdr(keyval_t *kv,
	       gvdr_sfducat_t *gc)
{
   if (get_ver(kv,gc))
      return 1;
   if (kc_kv_sscanf(keys_gvdr,kv,"PRODUCT_SEQUENCE_NUMBER",&gc->seqno))
      return 1;
   if (get_seqno_in(kv,gc))
      return 1;
   if (kc_kv_sscanf(keys_gvdr,kv,"HARDWARE_VERSION_ID_1",&gc->hw_ver1))
      return 1;
   if (kc_kv_sscanf(keys_gvdr,kv,"HARDWARE_VERSION_ID_2",&gc->hw_ver2))
      return 1;
   if (kc_kv_sscanf(keys_gvdr,kv,"HARDWARE_VERSION_ID_3",&gc->hw_ver3))
      return 1;
   if (kc_kv_sscanf(keys_gvdr,kv,"MAJOR_SOFTWARE_VERSION_ID",&gc->sw_ver1))
      return 1;
   if (kc_kv_sscanf(keys_gvdr,kv,"MINOR_SOFTWARE_VERSION_ID",&gc->sw_ver2))
      return 1;
   if (kc_kv_sscanf(keys_gvdr,kv,"MAP_PROJECTION_ID",&gc->proj))
      return 1;
   if (kc_kv_sscanf(keys_gvdr,kv,"MAP_HEMISPHERE_ID",&gc->hemi))
      return 1;
   gc->proc_time = kv_lookup(kv,"PROCESS_TIME");
   gc->data_fmt  = kv_lookup(kv,"DATA_FORMAT_TYPE");
   if (get_gfmt(gc))
      return 1;
   return 0;
}

static int
get_ver(keyval_t *kv,
	gvdr_sfducat_t *gc)
{
   char *p;

   if ((p=kv_lookup(kv,"PRODUCT_FILE_NAME"))==NULL)
      return 1;
   if (sscanf(p,"%*[^.].%ld",&gc->ver)!=1)
      return 2;
   return 0;
}

static int
get_seqno_in(keyval_t *kv,
	     gvdr_sfducat_t *gc)
{
   char *val,*buf,*p,*q;

   if ((val=kv_lookup(kv,"INPUT_SEQUENCE_NUMBERS"))==NULL)
      return 1;
   malloc_err(buf,strlen(val)+1);
   strcpy(buf,val);
   gc->Nseqno_in	= 0;
   gc->seqno_in		= NULL;
   if (p=strtok(buf,"(,)")) {
      while (p) {
	 gc->Nseqno_in++;
	 realloct_err(gc->seqno_in,gc->Nseqno_in,unsigned long);
	 gc->seqno_in[gc->Nseqno_in-1] = strtol(p,&q,10);
	 if (q==p || *q!='\0')
	    return 2;
	 p=strtok(NULL,"(,)");
      }
   }
   free(buf);
   return 0;
}

static int
get_gfmt(gvdr_sfducat_t *gc)
{
   if (!gc->data_fmt)
      return 1;
   if (!strcmp(gc->data_fmt,"IEEE")) {
      gc->gfmt_float=GFMT_IEEE;
      gc->gfmt_byte=GFMT_BIGEND;
   } else if (!strncmp(gc->data_fmt,"VAX",3)) {
      gc->gfmt_float=GFMT_VAX;
      gc->gfmt_byte=GFMT_LITEND;
   } else
      return 1;
   return 0;
}

/******************************************************************************
  gsfdu_rihdr

Reads the SFDU associated with the index.  Returns nonzero on failure.
******************************************************************************/

int
gsfdu_rihdr(FILE *fp,
	    int *len)
{
   sfdu_t sf;

   if (fread(&sf,sizeof(sfdu_t),1,fp)!=1) {
      error(0,errno,"[gsfdu_wihdr] fread failed");
      return 1;
   }
   if (strncmp(sf.ctrl,TYPE_GVDR_INDEX,strlen(TYPE_GVDR_INDEX)))
      return 2;
   *len=sf_len(&sf);
   return 0;
}

/******************************************************************************
  gsfdu_rlhdr

Reads the SFDU associated with a line (or tile).  Returns nonzero on failure.
******************************************************************************/

int
gsfdu_rlhdr(FILE *fp,
	    int *len)
{
   sfdu_t sf;

   if (fread(&sf,sizeof(sfdu_t),1,fp)!=1) {
      error(0,errno,"[gsfdu_wlhdr] fread failed");
      return 1;
   }
   if (strncmp(sf.ctrl,TYPE_GVDR_TILE,strlen(TYPE_GVDR_TILE)))
      return 2;
   *len=sf_len(&sf);
   return 0;
}

/******************************************************************************
  gsfdu_rtrl

Reads an SFDU trailer from the file fp.  Returns nonzero on failure.
******************************************************************************/

int
gsfdu_rtrl(FILE *fp)
{
   sfdu_t sf;
   keyval_t *kv=NULL;
   int err;

   err=gsfdu_rhdr2(fp,&sf,&kv);
   kv_free(kv);
   return err;
}

