/*
 *  GVDR library for reading GVDR data files
 *  Copyright (C) 1994 Michael J. Maurer
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Michael Maurer <maurer@nova.stanford.edu>
 *  Durand Bldg - Room 232
 *  Stanford, CA 94305-4055
 *  (415) 723-1024
 */
/******************************************************************************
  gvdr.h

This file is part of the STARLab Magellan Altimeter Data Processing Software.
Michael Maurer, March 1993.
******************************************************************************/
/* $Log: gvdr.h,v $
 * Revision 1.8  1994/03/04  22:05:02  maurer
 * Renamed some fields, added NEWANF for future change.
 *
 * Revision 1.7  1994/02/02  02:13:13  maurer
 * Changed ADF from min/max to mean/var.
 *
 * Revision 1.6  1994/01/06  00:13:04  maurer
 * Added ga_azim and gh_ANF_Nazim.
 *
 * Revision 1.5  1993/09/10  19:36:01  maurer
 * Added gc2 to gvdrhdr_t.
 *
 * Revision 1.4  1993/08/25  21:11:19  maurer
 * Added ADF record to GVDR cell.
 * Added reclen to GVDR cell.
 * Changed binary file header record to be parallel to PDS standards.
 *
 * Revision 1.3  1993/08/02  00:59:10  maurer
 * Removed map.h and util.h for public distribution.
 * Changed all uchar -> unsigned char, etc.
 *
 * Revision 1.2  1993/07/30  02:01:23  maurer
 * Removed SFDU from GVDR cells.
 * Added index definitions.
 *
 * Revision 1.1  1993/07/03  23:45:46  maurer
 * Changed GVDR record format:
 * 	moved NFF into ANF
 * 	reduced many sfloats to 1-byte
 * 	removed some other fields
 *
 * Revision 1.0  1993/06/12  00:37:27  maurer
 * Initial revision
 * */

#ifndef GVDR_H
#define GVDR_H

#include "sfdug.h"

#define GVDR_PAD '^'

/******************************************************************************
  GVDR data types

The gv_pack.c module converts between the compressed format GVDR stored
on tapes and the expanded format GVDR kept in memory.  Types ending with a
"_t" are expanded in-memory types, while those ending in "_rec" are
compressed on-tape types.
******************************************************************************/

/******************************************************************************
  gvdrxif_t
  gvdrxif_rec

Image statistics results from SIF and OIF files on SCVDR (Sinusoidal
Image File and Oblique sinusoidal Image File).
******************************************************************************/

typedef struct {
   unsigned short	gx_Nlooks;		/* # of observations used in averaging */
   float		gx_azim;		/* average azimuthal angle, degrees */
   float		gx_inc;			/* average incidence angle, degrees */
   float		gx_poln;		/* polarization angle, degrees */
   unsigned char	gx_hlo;			/* average SAR histogram lower knee, DNs */
   unsigned char	gx_hmed;		/* average SAR histogram median, DNs */
   unsigned char	gx_hhi;			/* average SAR histogram upper knee, DNs */
   unsigned char	gx_hpk;			/* average SAR histogram mode, DNs */
   float		gx_c0;			/* average SAR scattering law constant term, dB */
   float		gx_c1;			/* average SAR scattering law linear term, dB/deg */
   float		gx_c2;			/* average SAR scattering law quadratic term, dB/deg^2 */
} gvdrxif_t;

typedef struct {
   unsigned short	gxr_Nlooks;		/* # of observations used in averaging */
   unsigned short	gxr_azim;		/* average azimuthal angle, packed degrees */
   unsigned short	gxr_inc;		/* average incidence angle, packed degrees */
   unsigned char	gxr_poln;		/* polarization angle, packed degrees */
   unsigned char	gxr_hlo;		/* average SAR histogram lower knee, DNs */
   unsigned char	gxr_hmed;		/* average SAR histogram median, DNs */
   unsigned char	gxr_hhi;		/* average SAR histogram upper knee, DNs */
   unsigned char	gxr_hpk;		/* average SAR histogram mode, DNs */
   unsigned char	gxr_c0;			/* average SAR scattering law constant term, dB */
   unsigned char	gxr_c1;			/* average SAR scattering law linear term, dB/deg */
   unsigned char	gxr_c2;			/* average SAR scattering law quadratic term, dB/deg^2 */
} gvdrxif_rec;

/******************************************************************************
  gvdredf_t
  gvdredf_rec

Image statistics results from EDF files on SCVDR or RDF files on ARCDR,
(Emissivity Data File and Radiometry Data File respectively).
******************************************************************************/

typedef struct {
   unsigned short	ge_Nlooks;		/* # of observations used in averaging */
   float		ge_azim;		/* average azimuthal angle, degrees */
   float		ge_poln;		/* polarization angle, degrees */
   float		ge_inc;			/* average incidence angle, degrees */
   float		ge_emiss;		/* average emissivity */
   float		ge_emissv;		/* emissivity variance estimate */
} gvdredf_t;

typedef struct {
   unsigned short	ger_Nlooks;		/* # of observations used in averaging */
   unsigned short	ger_azim;		/* average azimuthal angle, packed degrees */
   unsigned short	ger_inc;		/* average incidence angle, packed degrees */
   unsigned char	ger_poln;		/* polarization angle, packed degrees */
   unsigned char	ger_emissv;		/* emissivity variance estimate */
   unsigned short	ger_emiss;		/* average emissivity, packed dimensionless */
} gvdredf_rec;

/******************************************************************************
  gvdradf_t
  gvdradf_rec

Altimeter results from ADF files on ARCDR.
******************************************************************************/

typedef struct {
   unsigned short	gd_Nlooks;		/* # of observations used in averaging */
   float		gd_radius;		/* mean radius observed */
   float		gd_radiusv;		/* unbiased sample variance of radius */
   float		gd_slope;		/* mean slope observed */
   float		gd_slopev;		/* unbiased sample variance of slope */
   float		gd_rho;			/* mean reflectivity observed */
   float		gd_rhov;		/* unbiased sample variance of reflectivity */
} gvdradf_t;

typedef struct {
   unsigned short	gdr_Nlooks;		/* # of observations used in averaging */
   unsigned short	gdr_radius;		/* mean radius observed */
   unsigned short	gdr_radiusv;		/* unbiased sample variance of radius */
   unsigned char	gdr_slope;		/* mean slope observed */
   unsigned char	gdr_slopev;		/* unbiased sample variance of slope */
   unsigned char	gdr_rho;		/* mean reflectivity observed */
   unsigned char	gdr_rhov;		/* unbiased sample variance of reflectivity */
} gvdradf_rec;

/******************************************************************************
  slaw_id

ID numbers for analytic scattering law types: Hagfors, Exponential,
Gaussian, Rayleigh, Muhleman.
******************************************************************************/

typedef enum {slHag,
	      slExp,
	      slGau,
	      slRay,
	      slMuh,
	      slNone} slaw_id;

/******************************************************************************
  gf_flags
  gfr_flags

Values for flags indicating problems with gvdrfit_t/gvdrfit_rec values.
******************************************************************************/

#define GFIT_P1MAX	0x01		/* P1 fit failed: P1 too large */
#define GFIT_P1MIN	0x02		/* P1 fit failed: P1 too small */
#define GFIT_SFIT	0x80		/* unknown error in sfit() */

/******************************************************************************
  gvdrfit_t
  gvdrfit_rec

For a single scattering law, results of least-squares best-fit to data in
gvdranf_t record.
******************************************************************************/

typedef struct {
   unsigned char	gf_slaw;		/* analytic scattering law ID number */
   unsigned char	gf_flags;		/* status flags for fit */
   float		gf_p1;			/* best-fit parameter 1 (shape) */
   float		gf_p1v;			/* estimated variance of parameter 1 */
   float		gf_p2;			/* best-fit parameter 2 (scale: Fresnel Reflectivity) */
   float		gf_p2v;			/* estimated variance of parameter 2*/
   float		gf_rms;			/* theoretical RMS slope of p1-p2 law, degrees */
   float		gf_rmsv;		/* estimated variance of RMS slope */
   float		gf_resid;		/* residual of fit */
} gvdrfit_t;

typedef struct {
   unsigned char	gfr_slaw;		/* analytic scattering law ID number */
   unsigned char	gfr_flags;		/* status flags for fit */
   unsigned char	gfr_p1;			/* best-fit parameter 1 (shape) */
   unsigned char	gfr_p1v;		/* estimated variance of parameter 1 */
   unsigned char	gfr_p2;			/* best-fit parameter 2 (scale: Fresnel Reflectivity) */
   unsigned char	gfr_p2v;		/* estimated variance of parameter 2*/
   unsigned char	gfr_rms;		/* theoretical RMS slope of p1-p2 law, degrees */
   unsigned char	gfr_rmsv;		/* estimated variance of RMS slope */
   unsigned char	gfr_resid;		/* residual of fit */
   unsigned char	gfr_pad;
} gvdrfit_rec;

/******************************************************************************
  gvdranf_t
  gvdranf_rec

Inverted scattering laws from ANF file on SCVDR (Altimetry Inversion File).
Best fit analytic scattering law coefficients.
******************************************************************************/

typedef struct {
   unsigned short	ga_Nlooks;		/* # of observations used in averaging */
   unsigned short	ga_Nsig0;		/* # of angles in scattering law */
   unsigned short	ga_NAsig0;
   unsigned short	ga_Nfit;		/* # of scattering law fits */
   unsigned short	ga_NAfit;
   float		ga_dcent;		/* average doppler centroid, Hz */
   float		ga_azim;		/* average ground track azimuth angle (def. depends on projection) */
   float		*ga_sig0;		/* average inverted scattering law, m^2/m^2 */
   float		*ga_Vsig0;		/* scattering law variance estimate */
   gvdrfit_t		*ga_fit;		/* array of scattering law fits */
} gvdranf_t;

#ifndef NEWANF
typedef struct {
   unsigned short	gar_reclen;		/* record length in bytes, not including this header */
   unsigned short	gar_Nlooks;		/* # of observations used in averaging */
   unsigned char	gar_Nsig0;		/* # of angles in scattering law */
   unsigned char	gar_Nfit;		/* # of scattering law fits */
   unsigned short	gar_dcent;		/* average doppler centroid, Hz */
   unsigned char	gar_azim;		/* average ground track azimuth angle (def. depends on projection) */
   unsigned char	gar_pad;
/* followed by:
   unsigned char	gar_sig0[gar_Nsig0];	/ * average inverted scattering law, m^2/m^2 * /
   unsigned char	gar_Vsig0[gar_Nsig0];	/ * scattering law variance estimate * /
   gvdrfit_rec		ga_fit[gar_fit];	/ * array of scattering law fits * /
*/
} gvdranf_rec;
#else
typedef struct {
   unsigned short	gar_reclen;		/* record length in bytes, not including this header */
   unsigned short	gar_Nlooks;		/* # of observations used in averaging */
   unsigned char	gar_Nsig0;		/* # of angles in scattering law */
   unsigned char	gar_Nfit;		/* # of scattering law fits */
   unsigned char	gar_dcent;		/* average doppler centroid, Hz */
   unsigned char	gar_azim;		/* average ground track azimuth angle (def. depends on projection) */
/* followed by:
   unsigned char	gar_sig0[gar_Nsig0];	/ * average inverted scattering law, m^2/m^2 * /
   unsigned char	gar_Vsig0[gar_Nsig0];	/ * scattering law variance estimate * /
   gvdrfit_rec		ga_fit[gar_fit];	/ * array of scattering law fits * /
*/
} gvdranf_rec;
#endif

/******************************************************************************
  gvdrcell_t
  gvdrcell_rec

Full data structure for a single pixel in the GVDR map projection.
******************************************************************************/

typedef struct {
   short		gv_X;			/* X-coordinate of pixel */
   short		gv_Y;			/* Y-coordinate of pixel */
   double		gv_lon;			/* longitude of center of cell, degrees */
   double		gv_lat;			/* latitude of center of cell, degrees */
   unsigned short	gv_reclen;		/* bytes this pixel occupies on disk, excluding reclen */
   unsigned short	gv_Nxif;		/* number of XIF data records to follow */
   unsigned short	gv_Nedf;		/* number of EDF/RDF data records to follow */
   unsigned short	gv_Nadf;		/* number of ADF data records to follow */
   unsigned short	gv_Nanf;		/* number of ANF data records to follow */
   unsigned short	gv_NAxif;		/* number of XIF data records allocated */
   unsigned short	gv_NAedf;		/* number of EDF/RDF data records allocated */
   unsigned short	gv_NAadf;		/* number of ADF data records allocated */
   unsigned short	gv_NAanf;		/* number of ANF data records allocated */
   unsigned short	gv_Msig0;		/* max number of angles in ANF scattering laws */
   unsigned short	gv_Mfit;		/* max number of scattering laws in NFF fits  */
   gvdrxif_t		*gv_xif;		/* array of XIF cohorts */
   gvdredf_t		*gv_edf;		/* array of EDF/RDF cohorts */
   gvdradf_t		*gv_adf;		/* array of ADF cohorts */
   gvdranf_t		*gv_anf;		/* array of ANF cohorts */
} gvdrcell_t;

typedef struct {
   unsigned short	gvr_reclen;		/* bytes this pixel occupies on disk, excluding reclen */
   unsigned char	gvr_Nxif;		/* number of XIF data records to follow */
   unsigned char	gvr_Nedf;		/* number of EDF/RDF data records to follow */
   unsigned char	gvr_Nadf;		/* number of ADF data records to follow */
   unsigned char	gvr_Nanf;		/* number of ANF data records to follow */
/* followed by:
   gvdrxif_rec		gvr_xif[gvr_Nxif];	/ * array of XIF cohorts * /
   gvdredf_rec		gvr_edf[gvr_Nedf];	/ * array of EDF/RDF cohorts * /
   gvdradf_rec		gvr_adf[gvr_Nadf];	/ * array of ADF cohorts * /
   gvdranf_rec		gvr_anf[gvr_Nanf];	/ * array of ANF cohorts * /
*/
} gvdrcell_rec;

/******************************************************************************
  gvdr_index_t, gvdr_index_rec

Data structure to maintain index of file pointer offsets for tiles,
lines and pixels.
******************************************************************************/

typedef struct {
   long			start;			/* file offset of beginning of index */
   unsigned short	mult;			/* multiplier for index values */
   short		lo;			/* index of first element in 'index' */
   short		hi;			/* index of last element in 'index' */
   unsigned short	nalloc;			/* bytes allocated to 'index' */
   unsigned short	nb;			/* bytes per element in 'index' */
   void			*index;			/* array of file offsets */
} gvdr_index_t;

typedef struct {
   short		lo;			/* index of first element in 'index' */
   short		hi;			/* index of last element in 'index' */
   unsigned short	nb;			/* bytes per element in 'index' */
   unsigned short	mult;
/* followed by (hi-lo+1) integers of nb bytes each.  nb={1,2,4}*/
} gvdr_index_rec;

/******************************************************************************
  gvdr_sfduhdr_t

GVDR file header binary header record.
If this is changed, be sure to change code in gv_cvt.c gcvt_hdr().
******************************************************************************/

typedef struct {
   unsigned long	gh_rec_tile;		/* tile/line record size, in bytes */
   unsigned long	gh_max_tlen;		/* largest tile/line size, in bytes */
   unsigned short	gh_rec_pixel;		/* pixel record size in bytes */
   unsigned short	gh_max_plen;		/* the largest pixel size, in bytes */
   unsigned short	gh_pixmax_XIF;		/* max number of XIF records in any pixel */
   unsigned short	gh_pixmax_EDF;		/* max number of EDF records in any pixel */
   unsigned short	gh_pixmax_ADF;		/* max number of ADF records in any pixel */
   unsigned short	gh_pixmax_ANF;		/* max number of ANF records in any pixel */
   unsigned short	gh_pixmax_Sig0;		/* max number of scattering angles any ANF pixel */
   unsigned short	gh_pixmax_Fit;		/* max number of scattering law fits in any ANF pixel */
   unsigned short	gh_tilemax_XIF;		/* max number of XIF records in any tile */
   unsigned short	gh_tilemax_EDF;		/* max number of EDF records in any tile */
   unsigned short	gh_tilemax_ADF;		/* max number of ADF records in any tile */
   unsigned short	gh_tilemax_ANF;		/* max number of ANF records in any tile */
   unsigned short	gh_XIF_Ninc;		/* number of cohorts XIF incidence angles are divided into */
   unsigned short	gh_XIF_Nazim;		/* number of cohorts XIF azimuthal angles are divided into */
   unsigned short	gh_EDF_Ninc;		/* number of cohorts EDF incidence angles are divided into */
   unsigned short	gh_EDF_Nazim;		/* number of cohorts EDF azimuthal angles are divided into */
   unsigned short	gh_ANF_Nazim;		/* number of cohorts ANF nadir-track azimuthal angles are divided into */
   unsigned short	gh_tile_ntx;		/* number of tiles in x direction */
   unsigned short	gh_tile_nty;		/* number of tiles in y direction */
   unsigned short	gh_tile_npx;		/* size of each tile in x direction */
   unsigned short	gh_tile_npy;		/* size of each tile in y direction */
   unsigned char	gh_proj_id;		/* map projection ID number */
   unsigned char	gh_proj_hemi;		/* map projection hemisphere ID number */
   long			gh_proj_bbxlo;		/* map projection x-coord of left edge of image bounding box */
   long			gh_proj_bbxhi;		/* map projection x-coord of rignt edge of image bounding box */
   long			gh_proj_bbylo;		/* map projection y-coord of bottom edge of image bounding box */
   long			gh_proj_bbyhi;		/* map projection y-coord of top edge of image bounding box */
   unsigned long	gh_proj_lines;		/* map projection number of lines in image (y samples) */
   unsigned long	gh_proj_samples;	/* map projection number of samples in image (x samples) */
   double		gh_proj_aradius;	/* map projection equatorial radius 1, km */
   double		gh_proj_bradius;	/* map projection equatorial radius 2, km */
   double		gh_proj_cradius;	/* map projection polar radius, km */
   double		gh_proj_stdpar1;	/* map projection first standard parallel, degrees */
   double		gh_proj_stdpar2;	/* map projection first standard parallel, degrees */
   double		gh_proj_cenlat;		/* map projection center latitude, degrees */
   double		gh_proj_cenlon;		/* map projection center longitude, degrees */
   unsigned long	gh_proj_linefirst;	/* map projection top line in image */
   unsigned long	gh_proj_linelast;	/* map projection bottom line in image */
   unsigned long	gh_proj_samplefirst;	/* map projection left pixel in each line */
   unsigned long	gh_proj_samplelast;	/* map projection right pixel in each line */
   double		gh_proj_rotation;	/* map projection rotation about center pixel, degrees clockwise */
   double		gh_proj_resolution;	/* map projection resolution, pixels/degree */
   double		gh_proj_scale;		/* map projection scale, degree/km */
   double		gh_proj_minlat;		/* map projection minimum latitude, degrees North */
   double		gh_proj_maxlat;		/* map projection maximum latitude, degrees North */
   double		gh_proj_minlon;		/* map projection minimum longitude, degrees East */
   double		gh_proj_maxlon;		/* map projection maximum longitude, degrees East */
   double		gh_proj_lineoffset;	/* map projection center in pixel coordinates: (1,1) is upper left */
   double		gh_proj_sampleoffset;	/* map projection center in pixel coordinates: (1,1) is upper left */
} gvdr_sfduhdr_t;

/******************************************************************************
  gvdr_sfducat_t

Contents of GVDR file header SFDU catalog.
******************************************************************************/

typedef struct {
   /* declare these as long so kc_kv_sscanf() works without "%hd" in keys_* */
   unsigned long	ver;
   unsigned long	seqno;
   unsigned long	Nseqno_in;
   unsigned long	*seqno_in;
   unsigned long	hw_ver1;
   unsigned long	hw_ver2;
   unsigned long	hw_ver3;
   unsigned long	sw_ver1;
   unsigned long	sw_ver2;
   unsigned long	proj;
   unsigned long	hemi;
   char			*proc_time;
   char			*data_fmt;
   int			gfmt_float;
   int			gfmt_byte;
} gvdr_sfducat_t;

/******************************************************************************
  gvdrhdr_t

Contents of entire GVDR file header.
******************************************************************************/

typedef struct {
   sfdu_t		sf1;
   sfdu_t		sf2;
   keyval_t		*kv2;
   gvdr_sfducat_t	gc2;
   sfdu_t		sf3;
   gvdr_sfduhdr_t	hdr3;
   sfdu_t		sf4;
   keyval_t		*kv4;
} gvdrhdr_t;


#endif /* GVDR_H */
