/*
 *  GVDR library for reading GVDR data files
 *  Copyright (C) 1994 Michael J. Maurer
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Michael Maurer <maurer@nova.stanford.edu>
 *  Durand Bldg - Room 232
 *  Stanford, CA 94305-4055
 *  (415) 723-1024
 */
static char rcsid[]="$Id: gvpdslib.c,v 1.2 1994/05/13 01:51:16 maurer Exp $";
/******************************************************************************
  gvpdslib.c

     Function:

This file is part of the STARLab Magellan Altimeter Data Processing Software.
Michael Maurer, March 1994.
******************************************************************************/
/* $Log: gvpdslib.c,v $
 * Revision 1.2  1994/05/13  01:51:16  maurer
 * Added gvfopen() to try to open files on CD-ROMs.
 *
 * Revision 1.1  1994/05/11  02:55:42  maurer
 * Slightly modified if tests in gvpds_rpixel().
 *
 * Revision 1.0  1994/04/29  02:06:21  maurer
 * Initial revision
 * */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "libmisc.h"
#include "gvdr.h"
#include "gvpdslib.h"
#include "pdstab.h"
#include "gvpdstab.h"	/* with static variables */
#include "gcvt.h"

#define GVPDSLIB_C
#include "gvpdslib.p"
#include "pdstab.p"
#include "gvdr.p"
#include "gcvt.p"

FILE *
gvfopen(char *fn,
	char *type)
{
   FILE *fp;
   char buf[1024],*p1,*p2;

   if (strlen(fn)+3>sizeof(buf))
      return NULL;

   /* try file name itself, with and without version ;1 */
   strcpy(buf,fn);
   if (fp=fopen(buf,type))
      return fp;
   strcat(buf,";1");
   if (fp=fopen(buf,type))
      return fp;

   /* try lower case file name, with and without version ;1 */
   for (p1=fn,p2=buf; *p1; p1++,p2++)
      *p2 = tolower(*p1);
   *p2 = 0;
   if (fp=fopen(buf,type))
      return fp;
   strcat(buf,";1");
   if (fp=fopen(buf,type))
      return fp;

   /* try upper case file name, with and without version ;1 */
   for (p1=fn,p2=buf; *p1; p1++,p2++)
      *p2 = toupper(*p1);
   *p2 = 0;
   if (fp=fopen(buf,type))
      return fp;
   strcat(buf,";1");
   if (fp=fopen(buf,type))
      return fp;

   return NULL;
}

void
gvdrhdr_fill(gvhdr_t *gvh,
	     gvdrhdr_t *hdr)
{
   memset(hdr,0,sizeof(gvdrhdr_t));
   hdr->hdr3.gh_pixmax_XIF		= gvh->gvh_pixmax_XIF;
   hdr->hdr3.gh_pixmax_EDF		= gvh->gvh_pixmax_EDF;
   hdr->hdr3.gh_pixmax_ADF		= gvh->gvh_pixmax_ADF;
   hdr->hdr3.gh_pixmax_ANF		= gvh->gvh_pixmax_ANF;
   hdr->hdr3.gh_pixmax_Sig0		= gvh->gvh_pixmax_Sig0;
   hdr->hdr3.gh_pixmax_Fit		= gvh->gvh_pixmax_Fit;
   hdr->hdr3.gh_tilemax_XIF		= gvh->gvh_tilemax_XIF;
   hdr->hdr3.gh_tilemax_EDF		= gvh->gvh_tilemax_EDF;
   hdr->hdr3.gh_tilemax_ADF		= gvh->gvh_tilemax_ADF;
   hdr->hdr3.gh_tilemax_ANF		= gvh->gvh_tilemax_ANF;
   hdr->hdr3.gh_XIF_Ninc		= gvh->gvh_XIF_Ninc;
   hdr->hdr3.gh_XIF_Nazim		= gvh->gvh_XIF_Nazim;
   hdr->hdr3.gh_EDF_Ninc		= gvh->gvh_EDF_Ninc;
   hdr->hdr3.gh_EDF_Nazim		= gvh->gvh_EDF_Nazim;
   hdr->hdr3.gh_ANF_Nazim		= gvh->gvh_ANF_Nazim;
   hdr->hdr3.gh_tile_ntx		= gvh->gvh_tile_ntx;
   hdr->hdr3.gh_tile_nty		= gvh->gvh_tile_nty;
   hdr->hdr3.gh_tile_npx		= gvh->gvh_tile_npx;
   hdr->hdr3.gh_tile_npy		= gvh->gvh_tile_npy;
   hdr->hdr3.gh_proj_id			= gvh->gvh_proj_id;
   hdr->hdr3.gh_proj_hemi		= gvh->gvh_proj_hemi;
   hdr->hdr3.gh_proj_bbxlo		= gvh->gvh_proj_bbxlo;
   hdr->hdr3.gh_proj_bbxhi		= gvh->gvh_proj_bbxhi;
   hdr->hdr3.gh_proj_bbylo		= gvh->gvh_proj_bbylo;
   hdr->hdr3.gh_proj_bbyhi		= gvh->gvh_proj_bbyhi;
   hdr->hdr3.gh_proj_lines		= gvh->gvh_proj_lines;
   hdr->hdr3.gh_proj_samples		= gvh->gvh_proj_samples;
   hdr->hdr3.gh_proj_aradius		= gvh->gvh_proj_aradius;
   hdr->hdr3.gh_proj_bradius		= gvh->gvh_proj_bradius;
   hdr->hdr3.gh_proj_cradius		= gvh->gvh_proj_cradius;
   hdr->hdr3.gh_proj_stdpar1		= gvh->gvh_proj_stdpar1;
   hdr->hdr3.gh_proj_stdpar2		= gvh->gvh_proj_stdpar2;
   hdr->hdr3.gh_proj_cenlat		= gvh->gvh_proj_cenlat;
   hdr->hdr3.gh_proj_cenlon		= gvh->gvh_proj_cenlon;
   hdr->hdr3.gh_proj_linefirst		= gvh->gvh_proj_linefirst;
   hdr->hdr3.gh_proj_linelast		= gvh->gvh_proj_linelast;
   hdr->hdr3.gh_proj_samplefirst	= gvh->gvh_proj_samplefirst;
   hdr->hdr3.gh_proj_samplelast		= gvh->gvh_proj_samplelast;
   hdr->hdr3.gh_proj_rotation		= gvh->gvh_proj_rotation;
   hdr->hdr3.gh_proj_resolution		= gvh->gvh_proj_resolution;
   hdr->hdr3.gh_proj_scale		= gvh->gvh_proj_scale;
   hdr->hdr3.gh_proj_minlat		= gvh->gvh_proj_minlat;
   hdr->hdr3.gh_proj_maxlat		= gvh->gvh_proj_maxlat;
   hdr->hdr3.gh_proj_minlon		= gvh->gvh_proj_minlon;
   hdr->hdr3.gh_proj_maxlon		= gvh->gvh_proj_maxlon;
   hdr->hdr3.gh_proj_lineoffset		= gvh->gvh_proj_lineoffset;
   hdr->hdr3.gh_proj_sampleoffset	= gvh->gvh_proj_sampleoffset;
}

void
gcvt_pidx(gpds_pidx_t *src,
	  gpds_pidx_t *dst,
	  unsigned int n)
{
   while (n--) {
      gcvt_s(&src->xif,&dst->xif,4);			/* xif...anf */
      src++;
      dst++;
   }
}

void
gcvt_tidx(gpds_tidx_t *src,
	  gpds_tidx_t *dst,
	  unsigned int n)
{
   gcvt_l(src,dst,n*sizeof(gpds_tidx_t)/sizeof(unsigned long));	/* all fields are type long */
}

void
gvpds_alloc(gvhdr_t *gvh,
	    gvdrcell_t *gv)
{
   gvdr_clear(gv);
   gvdr_realloc(gv,
		gvh->gvh_pixmax_XIF,
		gvh->gvh_pixmax_EDF,
		gvh->gvh_pixmax_ADF,
		gvh->gvh_pixmax_ANF,
		gvh->gvh_pixmax_Sig0,
		gvh->gvh_pixmax_Fit);
}

void
gvpds_t2p(gvhdr_t *gvh,
	  int tx,
	  int ty,
	  int px,
	  int py,
	  int *x,
	  int *y)
{
   int x0,y0,npx,npy;

   x0  = -gvh->gvh_proj_sampleoffset;
   y0  =  gvh->gvh_proj_lineoffset;
   npx =  gvh->gvh_tile_npx;
   npy =  gvh->gvh_tile_npy;
   *x = x0 + npx*tx + px;
   *y = y0 - npy*ty - py;
}

void
gvpds_p2t(gvhdr_t *gvh,
	  int x,
	  int y,
	  int *tx,
	  int *ty,
	  int *px,
	  int *py)
{
   int x0,y0,npx,npy;

   x0  = -gvh->gvh_proj_sampleoffset;
   y0  =  gvh->gvh_proj_lineoffset;
   npx =  gvh->gvh_tile_npx;
   npy =  gvh->gvh_tile_npy;
   x -= x0;
   y -= y0;
   y = -y;
   *tx = x/npx;
   *ty = y/npy;
   *px = x-*tx*npx;
   *py = y-*ty*npy;
}

int
gvhdr_read(FILE *fp,
	   gvhdr_t *gvh)
{
   return pdstable_readrow(fp,pt_gvhdr,gvh);
}

gpds_tidx_t *
gvtidx_alloc(gvhdr_t *gvh)
{
   size_t ntil;

   ntil = gvh->gvh_tile_ntx * gvh->gvh_tile_nty;
   return (gpds_tidx_t *)malloc(ntil*sizeof(gpds_tidx_t));
}
	     
int
gvtidx_read(FILE *fp,
	    gvhdr_t *gvh,
	    gpds_tidx_t *tidx)
{
   size_t ntil;

   ntil = gvh->gvh_tile_ntx * gvh->gvh_tile_nty;
   if (fread(tidx,sizeof(gpds_tidx_t),ntil,fp)!=ntil)
      return 1;
   gcvt_setmode(GCVT_READ,gvh->gvh_fmt_float,gvh->gvh_fmt_byte);
   gcvt_tidx(tidx,tidx,ntil);
   return 0;
}

gpds_pidx_t *
gvpidx_alloc(gvhdr_t *gvh)
{
   size_t npix;

   npix = gvh->gvh_tile_npx * gvh->gvh_tile_npx;
   return (gpds_pidx_t *)malloc(npix*sizeof(gpds_pidx_t));
}
	     
int
gvpidx_rtile(FILE *fp,
	     gvhdr_t *gvh,
	     gpds_tidx_t *tidx,
	     int tx,
	     int ty,
	     gpds_pidx_t *pidx)
{
   size_t i,npix;
   fpos_t fpos;

   if (tx<0 || tx>=gvh->gvh_tile_ntx ||
       ty<0 || ty>=gvh->gvh_tile_nty)
      return 1;
   i = tx + ty*gvh->gvh_tile_ntx;
   fpos = tidx[i].pix * sizeof(gpds_pidx_t);
   if (0) printf("pidx i=%d pix=%d fpos=%d\n",i,(int)tidx[i].pix,(int)fpos);
   if (fsetpos(fp,&fpos))
      return 2;
   npix = gvh->gvh_tile_npx * gvh->gvh_tile_npy;
   if (fread(pidx,sizeof(gpds_pidx_t),npix,fp)!=npix)
      return 3;
   gcvt_setmode(GCVT_READ,gvh->gvh_fmt_float,gvh->gvh_fmt_byte);
   gcvt_pidx(pidx,pidx,npix);
   return 0;
}

int
gvpidx_rpixel(gvhdr_t *gvh,
	      gpds_pidx_t *pidx,
	      int px,
	      int py,
	      gpds_pidx_t *P)
{
   size_t i;

   if (px<0 || px>=gvh->gvh_tile_npx ||
       py<0 || py>=gvh->gvh_tile_npy)
      return 1;
   i = px + py*gvh->gvh_tile_npx;
   *P = pidx[i];
   return 0;
}	      

gvdrxif_rec *
gvxif_alloc(gvhdr_t *gvh)
{
   size_t npix;

   npix = gvh->gvh_tilemax_XIF;
   return (gvdrxif_rec *)malloc(npix*sizeof(gvdrxif_rec));
}
	     
int
gvxif_rtile(FILE *fp,
	    gvhdr_t *gvh,
	    gpds_tidx_t *tidx,
	    int tx,
	    int ty,
	    gvdrxif_rec *xif)
{
   size_t i,npix;
   fpos_t fpos;

   if (tx<0 || tx>=gvh->gvh_tile_ntx ||
       ty<0 || ty>=gvh->gvh_tile_nty)
      return 1;
   i = tx + ty*gvh->gvh_tile_ntx;
   fpos = tidx[i].xif * sizeof(gvdrxif_rec);
   if (fsetpos(fp,&fpos))
      return 2;
   npix = tidx[i].Nxif;
   if (fread(xif,sizeof(gvdrxif_rec),npix,fp)!=npix)
      return 3;
   return 0;
}

int
gvxif_rpixel(gvhdr_t *gvh,
	     gpds_pidx_t *pidx,
	     gvdrxif_rec *xif,
	     int px,
	     int py,
	     gvdrxif_t *P)
{
   size_t np,nx;
   fpos_t fpos;
   char *p;

   if (px<0 || px>=gvh->gvh_tile_npx ||
       py<0 || py>=gvh->gvh_tile_npy)
      return 1;
   np = px + py*gvh->gvh_tile_npx;
   nx = pidx[np].xif;
   p = (char *)&xif[nx];
   return gvdr_mrxif(&p,P,pidx[np].Nxif);
}	      

gvdredf_rec *
gvedf_alloc(gvhdr_t *gvh)
{
   size_t npix;

   npix = gvh->gvh_tilemax_EDF;
   return (gvdredf_rec *)malloc(npix*sizeof(gvdredf_rec));
}
	     
int
gvedf_rtile(FILE *fp,
	    gvhdr_t *gvh,
	    gpds_tidx_t *tidx,
	    int tx,
	    int ty,
	    gvdredf_rec *edf)
{
   size_t i,npix;
   fpos_t fpos;

   if (tx<0 || tx>=gvh->gvh_tile_ntx ||
       ty<0 || ty>=gvh->gvh_tile_nty)
      return 1;
   i = tx + ty*gvh->gvh_tile_ntx;
   fpos = tidx[i].edf * sizeof(gvdredf_rec);
   if (fsetpos(fp,&fpos))
      return 2;
   npix = tidx[i].Nedf;
   if (fread(edf,sizeof(gvdredf_rec),npix,fp)!=npix)
      return 3;
   return 0;
}

int
gvedf_rpixel(gvhdr_t *gvh,
	     gpds_pidx_t *pidx,
	     gvdredf_rec *edf,
	     int px,
	     int py,
	     gvdredf_t *P)
{
   size_t np,nx;
   fpos_t fpos;
   char *p;

   if (px<0 || px>=gvh->gvh_tile_npx ||
       py<0 || py>=gvh->gvh_tile_npy)
      return 1;
   np = px + py*gvh->gvh_tile_npx;
   nx = pidx[np].edf;
   p = (char *)&edf[nx];
   return gvdr_mredf(&p,P,pidx[np].Nedf);
}	      

gvdradf_rec *
gvadf_alloc(gvhdr_t *gvh)
{
   size_t npix;

   npix = gvh->gvh_tilemax_ADF;
   return (gvdradf_rec *)malloc(npix*sizeof(gvdradf_rec));
}
	     
int
gvadf_rtile(FILE *fp,
	    gvhdr_t *gvh,
	    gpds_tidx_t *tidx,
	    int tx,
	    int ty,
	    gvdradf_rec *adf)
{
   size_t i,npix;
   fpos_t fpos;

   if (tx<0 || tx>=gvh->gvh_tile_ntx ||
       ty<0 || ty>=gvh->gvh_tile_nty)
      return 1;
   i = tx + ty*gvh->gvh_tile_ntx;
   fpos = tidx[i].adf * sizeof(gvdradf_rec);
   if (fsetpos(fp,&fpos))
      return 2;
   npix = tidx[i].Nadf;
   if (fread(adf,sizeof(gvdradf_rec),npix,fp)!=npix)
      return 3;
   return 0;
}

int
gvadf_rpixel(gvhdr_t *gvh,
	     gpds_pidx_t *pidx,
	     gvdradf_rec *adf,
	     int px,
	     int py,
	     gvdradf_t *P)
{
   size_t np,nx;
   fpos_t fpos;
   char *p;

   if (px<0 || px>=gvh->gvh_tile_npx ||
       py<0 || py>=gvh->gvh_tile_npy)
      return 1;
   np = px + py*gvh->gvh_tile_npx;
   nx = pidx[np].adf;
   p = (char *)&adf[nx];
   return gvdr_mradf(&p,P,pidx[np].Nadf);
}	      

void *
gvanf_alloc(gvhdr_t *gvh)
{
   size_t npix,reclen;

   npix = gvh->gvh_tilemax_ANF;
   reclen = gvh->gvh_reclen_ANF;
   return malloc(npix*reclen);
}
	     
int
gvanf_rtile(FILE *fp,
	    gvhdr_t *gvh,
	    gpds_tidx_t *tidx,
	    int tx,
	    int ty,
	    void *anf)
{
   size_t i,npix,reclen;
   fpos_t fpos;

   if (tx<0 || tx>=gvh->gvh_tile_ntx ||
       ty<0 || ty>=gvh->gvh_tile_nty)
      return 1;
   i = tx + ty*gvh->gvh_tile_ntx;
   reclen = gvh->gvh_reclen_ANF;
   fpos = tidx[i].anf * reclen;
   if (fsetpos(fp,&fpos))
      return 2;
   npix = tidx[i].Nanf;
   if (fread(anf,reclen,npix,fp)!=npix)
      return 3;
   return 0;
}

int
gvanf_rpixel(gvhdr_t *gvh,
	     gpds_pidx_t *pidx,
	     void *anf,
	     int px,
	     int py,
	     gvdranf_t *P)
{
   size_t np,nx,reclen;
   fpos_t fpos;
   char *p;

   if (px<0 || px>=gvh->gvh_tile_npx ||
       py<0 || py>=gvh->gvh_tile_npy)
      return 1;
   np = px + py*gvh->gvh_tile_npx;
   nx = pidx[np].anf;
   reclen = gvh->gvh_reclen_ANF;
   p = (char *)anf + nx*reclen;;
   return gvdr_mranf(&p,P,pidx[np].Nanf,gvh->gvh_pixmax_Sig0);
}	      

int
gvpds_rpixel(gvhdr_t *gvh,
	     gpds_pidx_t *pidx,
	     gvdrxif_rec *xif,
	     gvdredf_rec *edf,
	     gvdradf_rec *adf,
	     gvdranf_rec *anf,
	     int px,
	     int py,
	     gvdrcell_t *gv)
{
   int err=0;
   size_t np;

   gvdr_reset(gv);
   if (pidx) {
      np = px + py*gvh->gvh_tile_npx;
      gv->gv_Nxif = pidx[np].Nxif;
      gv->gv_Nedf = pidx[np].Nedf;
      gv->gv_Nadf = pidx[np].Nadf;
      gv->gv_Nanf = pidx[np].Nanf;
      if (xif) err |= gvxif_rpixel(gvh,pidx,xif,px,py,gv->gv_xif);
      if (edf) err |= gvedf_rpixel(gvh,pidx,edf,px,py,gv->gv_edf);
      if (adf) err |= gvadf_rpixel(gvh,pidx,adf,px,py,gv->gv_adf);
      if (anf) err |= gvanf_rpixel(gvh,pidx,anf,px,py,gv->gv_anf);
   }
   return err;
}
