/*
 *  GVDR library for reading GVDR data files
 *  Copyright (C) 1994 Michael J. Maurer
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Michael Maurer <maurer@nova.stanford.edu>
 *  Durand Bldg - Room 232
 *  Stanford, CA 94305-4055
 *  (415) 723-1024
 */
/******************************************************************************
  gvpdstab.h

This file is part of the STARLab Magellan Altimeter Data Processing Software.
Michael Maurer, March 1994.
******************************************************************************/
/* $Log: gvpdstab.h,v $
 * Revision 1.1  1994/05/11  00:13:42  maurer
 * Split pathname in gpds_index_t into two components.
 * Changed some field widths to have even line lengths.
 *
 * Revision 1.0  1994/04/29  02:07:00  maurer
 * Initial revision
 * */

#ifndef GVPDSTAB_H
#define GVPDSTAB_H

#include <stddef.h>	/* for offsetof() */
#include "pdstab.h"

typedef struct {
   unsigned short	gvh_hw_ver1;		/* hardware version ID 1 */
   unsigned short	gvh_hw_ver2;		/* hardware version ID 2 */
   unsigned short	gvh_hw_ver3;		/* hardware version ID 3 */
   unsigned short	gvh_sw_ver1;		/* software version ID 1 */
   unsigned short	gvh_sw_ver2;		/* software version ID 2 */
   unsigned short	gvh_fmt_float;		/* floating point format ID (see gcvt.h) */
   unsigned short	gvh_fmt_byte;		/* byte-order ID (see gcvt.h) */
   unsigned short	gvh_pixmax_XIF;		/* max number of XIF records in any pixel */
   unsigned short	gvh_pixmax_EDF;		/* max number of EDF records in any pixel */
   unsigned short	gvh_pixmax_ADF;		/* max number of ADF records in any pixel */
   unsigned short	gvh_pixmax_ANF;		/* max number of ANF records in any pixel */
   unsigned short	gvh_pixmax_Sig0;	/* max number of scattering angles any ANF pixel */
   unsigned short	gvh_pixmax_Fit;		/* max number of scattering law fits in any ANF pixel */
   unsigned short	gvh_tilemax_XIF;	/* max number of XIF records in any tile */
   unsigned short	gvh_tilemax_EDF;	/* max number of EDF records in any tile */
   unsigned short	gvh_tilemax_ADF;	/* max number of ADF records in any tile */
   unsigned short	gvh_tilemax_ANF;	/* max number of ANF records in any tile */
   unsigned short	gvh_reclen_ANF;		/* ANF PDS table record length (bytes per row) */
   unsigned short	gvh_XIF_Ninc;		/* number of cohorts XIF incidence angles are divided into */
   unsigned short	gvh_XIF_Nazim;		/* number of cohorts XIF azimuthal angles are divided into */
   unsigned short	gvh_EDF_Ninc;		/* number of cohorts EDF incidence angles are divided into */
   unsigned short	gvh_EDF_Nazim;		/* number of cohorts EDF azimuthal angles are divided into */
   unsigned short	gvh_ANF_Nazim;		/* number of cohorts ANF nadir-track azimuthal angles are divided into */
   unsigned short	gvh_tile_ntx;		/* number of tiles in x direction */
   unsigned short	gvh_tile_nty;		/* number of tiles in y direction */
   unsigned short	gvh_tile_npx;		/* size of each tile in x direction */
   unsigned short	gvh_tile_npy;		/* size of each tile in y direction */
   unsigned short	gvh_proj_id;		/* map projection ID number */
   unsigned short	gvh_proj_hemi;		/* map projection hemisphere ID number */
   long			gvh_proj_bbxlo;		/* map projection x-coord of left edge of image bounding box */
   long			gvh_proj_bbxhi;		/* map projection x-coord of rignt edge of image bounding box */
   long			gvh_proj_bbylo;		/* map projection y-coord of bottom edge of image bounding box */
   long			gvh_proj_bbyhi;		/* map projection y-coord of top edge of image bounding box */
   unsigned int		gvh_proj_lines;		/* map projection number of lines in image (y samples) */
   unsigned int		gvh_proj_samples;	/* map projection number of samples in image (x samples) */
   double		gvh_proj_aradius;	/* map projection equatorial radius 1, km */
   double		gvh_proj_bradius;	/* map projection equatorial radius 2, km */
   double		gvh_proj_cradius;	/* map projection polar radius, km */
   double		gvh_proj_stdpar1;	/* map projection first standard parallel, degrees */
   double		gvh_proj_stdpar2;	/* map projection first standard parallel, degrees */
   double		gvh_proj_cenlat;	/* map projection center latitude, degrees */
   double		gvh_proj_cenlon;	/* map projection center longitude, degrees */
   unsigned int		gvh_proj_linefirst;	/* map projection top line in image */
   unsigned int		gvh_proj_linelast;	/* map projection bottom line in image */
   unsigned int		gvh_proj_samplefirst;	/* map projection left pixel in each line */
   unsigned int		gvh_proj_samplelast;	/* map projection right pixel in each line */
   double		gvh_proj_rotation;	/* map projection rotation about center pixel, degrees clockwise */
   double		gvh_proj_resolution;	/* map projection resolution, pixels/degree */
   double		gvh_proj_scale;		/* map projection scale, degree/km */
   double		gvh_proj_minlat;	/* map projection minimum latitude, degrees North */
   double		gvh_proj_maxlat;	/* map projection maximum latitude, degrees North */
   double		gvh_proj_minlon;	/* map projection minimum longitude, degrees East */
   double		gvh_proj_maxlon;	/* map projection maximum longitude, degrees East */
   double		gvh_proj_lineoffset;	/* map projection center in pixel coordinates: (1,1) is upper left */
   double		gvh_proj_sampleoffset;	/* map projection center in pixel coordinates: (1,1) is upper left */
} gvhdr_t;

typedef struct {
   char		volume_id[20];
   char		data_set_id[40];
   char		product_id[40];
   char		gvdr_dir_name[40];
   char		gvdr_file_name[40];
} gpds_index_t;

typedef struct {
   char			source_file_name[12];
   unsigned long	total_footprint;
   unsigned long	accepted_footprint;
   unsigned long	rejected_footprint;
} gpds_gvorb_t;

typedef struct {
   int		tile_coord_x;
   int		tile_coord_y;
   float	top_left_longitude;
   float	top_left_latitude;
   float	top_right_longitude;
   float	top_right_latitude;
   float	bottom_left_longitude;
   float	bottom_left_latitude;
   float	bottom_right_longitude;
   float	bottom_right_latitude;
} gpds_tgeom_t;

typedef struct {
   float	longitude;
   float	latitude;
   int		image_coord_x;
   int		image_coord_y;
   int		tile_coord_x;
   int		tile_coord_y;
   int		pixel_coord_x;
   int		pixel_coord_y;
} gvgeo_t;

typedef struct {
   gvgeo_t	top_left;
   gvgeo_t	top_right;
   gvgeo_t	bottom_left;
   gvgeo_t	bottom_right;
   char		feature_type[256];
   char		feature_name[256];
   char		feature_diac[256];
   char		iau_status[256];
} gpds_gvgeo_t;

typedef struct {
   unsigned long	pix;
   unsigned long	xif;
   unsigned long	edf;
   unsigned long	adf;
   unsigned long	anf;
   unsigned long	Nxif;
   unsigned long	Nedf;
   unsigned long	Nadf;
   unsigned long	Nanf;
} gpds_tidx_t;

typedef struct {
   unsigned short	xif;
   unsigned short	edf;
   unsigned short	adf;
   unsigned short	anf;
   unsigned char	Nxif;
   unsigned char	Nedf;
   unsigned char	Nadf;
   unsigned char	Nanf;
} gpds_pidx_t;

static pdstable_t pt_gvhdr[] = {
   "HARDWARE_VERSION_ID_1",	"%2d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_hw_ver1),		0, 0,
   "HARDWARE_VERSION_ID_2",	"%2d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_hw_ver2),		0, 0,
   "HARDWARE_VERSION_ID_3",	"%2d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_hw_ver3),		0, 0,
   "SOFTWARE_VERSION_ID_1",	"%2d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_sw_ver1),		0, 0,
   "SOFTWARE_VERSION_ID_2",	"%2d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_sw_ver2),		0, 0,
   "FLOAT_FORMAT",		"%2d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_fmt_float),	0, 0,
   "BYTE_FORMAT",		"%2d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_fmt_byte),		0, 0,
   "XIF_SAMPLES_MAXIMUM",	"%3d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_pixmax_XIF),	0, 0,
   "RDF_SAMPLES_MAXIMUM",	"%3d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_pixmax_EDF),	0, 0,
   "ADF_SAMPLES_MAXIMUM",	"%3d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_pixmax_ADF),	0, 0,
   "ANF_SAMPLES_MAXIMUM",	"%3d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_pixmax_ANF),	0, 0,
   "SCATTERING_ANGLE_MAXIMUM",	"%3d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_pixmax_Sig0),	0, 0,
   "SCATTERING_FIT_MAXIMUM",	"%3d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_pixmax_Fit),	0, 0,
   "XIF_TILE_SAMPLES_MAXIMUM",	"%5d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_tilemax_XIF),	0, 0,
   "RDF_TILE_SAMPLES_MAXIMUM",	"%5d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_tilemax_EDF),	0, 0,
   "ADF_TILE_SAMPLES_MAXIMUM",	"%5d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_tilemax_ADF),	0, 0,
   "ANF_TILE_SAMPLES_MAXIMUM",	"%5d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_tilemax_ANF),	0, 0,
   "ANF_RECORD_BYTES",		"%5d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_reclen_ANF),	0, 0,
   "XIF_COHORT_INCIDENCE_COUNT","%3d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_XIF_Ninc),		0, 0,
   "XIF_COHORT_AZIMUTH_COUNT",	"%3d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_XIF_Nazim),	0, 0,
   "RDF_COHORT_INCIDENCE_COUNT","%3d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_EDF_Ninc),		0, 0,
   "RDF_COHORT_AZIMUTH_COUNT",	"%3d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_EDF_Nazim),	0, 0,
   "ANF_COHORT_AZIMUTH_COUNT",	"%3d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_ANF_Nazim),	0, 0,
   "HORIZONTAL_TILE_COUNT",	"%3d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_tile_ntx),		0, 0,
   "VERTICAL_TILE_COUNT",	"%3d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_tile_nty),		0, 0,
   "HORIZONTAL_TILE_SIZE",	"%4d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_tile_npx),		0, 0,
   "VERTICAL_TILE_SIZE",	"%4d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_tile_npy),		0, 0,
   "MAP_PROJECTION_ID_1",	"%2d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_proj_id),		0, 0,
   "MAP_PROJECTION_ID_2",	"%2d",		"%hu",	T_USHORT,	offsetof(gvhdr_t,gvh_proj_hemi),	0, 0,
   "LEFTMOST_MAP_COORD",	"%5d",		"%d",	T_INT,		offsetof(gvhdr_t,gvh_proj_bbxlo),	0, 0,
   "RIGHTMOST_MAP_COORD",	"%5d",		"%d",	T_INT,		offsetof(gvhdr_t,gvh_proj_bbxhi),	0, 0,
   "BOTTOMMOST_MAP_COORD",	"%5d",		"%d",	T_INT,		offsetof(gvhdr_t,gvh_proj_bbylo),	0, 0,
   "TOPMOST_MAP_COORD",		"%5d",		"%d",	T_INT,		offsetof(gvhdr_t,gvh_proj_bbyhi),	0, 0,
   "PROJECTION_LINES",		"%5d",		"%u",	T_UINT,		offsetof(gvhdr_t,gvh_proj_lines),	0, 0,
   "PROJECTION_SAMPLES",	"%5d",		"%u",	T_UINT,		offsetof(gvhdr_t,gvh_proj_samples),	0, 0,
   "A_AXIS_RADIUS",		"%12.2f",	"%lf",	T_DOUBLE,	offsetof(gvhdr_t,gvh_proj_aradius),	0, 0,
   "B_AXIS_RADIUS",		"%12.2f",	"%lf",	T_DOUBLE,	offsetof(gvhdr_t,gvh_proj_bradius),	0, 0,
   "C_AXIS_RADIUS",		"%12.2f",	"%lf",	T_DOUBLE,	offsetof(gvhdr_t,gvh_proj_cradius),	0, 0,
   "FIRST_STANDARD_PARALLEL",	"%10.5f",	"%lf",	T_DOUBLE,	offsetof(gvhdr_t,gvh_proj_stdpar1),	0, 0,
   "SECOND_STANDARD_PARALLEL",	"%10.5f",	"%lf",	T_DOUBLE,	offsetof(gvhdr_t,gvh_proj_stdpar2),	0, 0,
   "CENTER_LATITUDE",		"%10.5f",	"%lf",	T_DOUBLE,	offsetof(gvhdr_t,gvh_proj_cenlat),	0, 0,
   "CENTER_LONGITUDE",		"%10.5f",	"%lf",	T_DOUBLE,	offsetof(gvhdr_t,gvh_proj_cenlon),	0, 0,
   "LINE_FIRST_PIXEL",		"%5d",		"%u",	T_UINT,		offsetof(gvhdr_t,gvh_proj_linefirst),	0, 0,
   "LINE_LAST_PIXEL",		"%5d",		"%u",	T_UINT,		offsetof(gvhdr_t,gvh_proj_linelast),	0, 0,
   "SAMPLE_FIRST_PIXEL",	"%5d",		"%u",	T_UINT,		offsetof(gvhdr_t,gvh_proj_samplefirst),	0, 0,
   "SAMPLE_LAST_PIXEL",		"%5d",		"%u",	T_UINT,		offsetof(gvhdr_t,gvh_proj_samplelast),	0, 0,
   "MAP_PROJECTION_ROTATION",	"%10.5f",	"%lf",	T_DOUBLE,	offsetof(gvhdr_t,gvh_proj_rotation),	0, 0,
   "MAP_RESOLUTION",		"%10.5f",	"%lf",	T_DOUBLE,	offsetof(gvhdr_t,gvh_proj_resolution),	0, 0,
   "MAP_SCALE",			"%10.5f",	"%lf",	T_DOUBLE,	offsetof(gvhdr_t,gvh_proj_scale),	0, 0,
   "MINIMUM_LATITUDE",		"%10.5f",	"%lf",	T_DOUBLE,	offsetof(gvhdr_t,gvh_proj_minlat),	0, 0,
   "MAXIMUM_LATITUDE",		"%10.5f",	"%lf",	T_DOUBLE,	offsetof(gvhdr_t,gvh_proj_maxlat),	0, 0,
   "WESTERNMOST_LONGITUDE",	"%10.5f",	"%lf",	T_DOUBLE,	offsetof(gvhdr_t,gvh_proj_minlon),	0, 0,
   "EASTERNMOST_LONGITUDE",	"%10.5f",	"%lf",	T_DOUBLE,	offsetof(gvhdr_t,gvh_proj_maxlon),	0, 0,
   "LINE_PROJECTION_OFFSET",	"%10.5f",	"%lf",	T_DOUBLE,	offsetof(gvhdr_t,gvh_proj_lineoffset),	0, 0,
   "SAMPLE_PROJECTION_OFFSET",	"%10.5f",	"%lf",	T_DOUBLE,	offsetof(gvhdr_t,gvh_proj_sampleoffset),0, 0,
   NULL,			NULL,		NULL,	0,		0,					0, 0
};

static pdstable_t pt_index[] = {
   "VOLUME_ID",			"\"%-9s\"",	"\"%.9s",	T_STRING,	offsetof(gpds_index_t,volume_id),		0, 0,
   "DATA_SET_ID",		"\"%-22s\"",	"\"%.22s",	T_STRING,	offsetof(gpds_index_t,data_set_id),		0, 0,
   "PRODUCT_ID",		"\"%-18s\"",	"\"%.18s",	T_STRING,	offsetof(gpds_index_t,product_id),		0, 0,
   "GVDR_DIRECTORY_NAME",	"\"%-13s\"",	"\"%.13s",	T_STRING,	offsetof(gpds_index_t,gvdr_dir_name),		0, 0,
   "GVDR_FILE_NAME",		"\"%-12s\"",	"\"%.12s",	T_STRING,	offsetof(gpds_index_t,gvdr_file_name),		0, 0,
   NULL,			NULL,		NULL,		0,		0,						0, 0
};

static pdstable_t pt_gvorb[] = {
   "SOURCE_FILE_NAME",		"\"%-11s\"",	"\"%.11s",	T_STRING,	offsetof(gpds_gvorb_t,source_file_name),	0, 0,
   "TOTAL_FOOTPRINT_COUNT",	"%6d",		"%lu",		T_ULONG,	offsetof(gpds_gvorb_t,total_footprint),		0, 0,
   "ACCEPTED_FOOTPRINT_COUNT",	"%6d",		"%lu",		T_ULONG,	offsetof(gpds_gvorb_t,accepted_footprint),	0, 0,
   "REJECTED_FOOTPRINT_COUNT",	"%6d",		"%lu",		T_ULONG,	offsetof(gpds_gvorb_t,rejected_footprint),	0, 0,
   NULL,			NULL,		NULL,		0,		0,						0, 0
};

static pdstable_t pt_tgeom[] = {
   "TILE_COORD_X",		"%5d",		"%d",		T_INT,		offsetof(gpds_tgeom_t,tile_coord_x),		0, 0,
   "TILE_COORD_Y",		"%4d",		"%d",		T_INT,		offsetof(gpds_tgeom_t,tile_coord_y),		0, 0,
   "TOP_LEFT_LONGITUDE",	"%8.3f",	"%f",		T_FLOAT,	offsetof(gpds_tgeom_t,top_left_longitude),	0, 0,
   "TOP_LEFT_LATITUDE",		"%8.3f",	"%f",		T_FLOAT,	offsetof(gpds_tgeom_t,top_left_latitude),	0, 0,
   "TOP_RIGHT_LONGITUDE",	"%8.3f",	"%f",		T_FLOAT,	offsetof(gpds_tgeom_t,top_right_longitude),	0, 0,
   "TOP_RIGHT_LATITUDE",	"%8.3f",	"%f",		T_FLOAT,	offsetof(gpds_tgeom_t,top_right_latitude),	0, 0,
   "BOTTOM_LEFT_LONGITUDE",	"%8.3f",	"%f",		T_FLOAT,	offsetof(gpds_tgeom_t,bottom_left_longitude),	0, 0,
   "BOTTOM_LEFT_LATITUDE",	"%8.3f",	"%f",		T_FLOAT,	offsetof(gpds_tgeom_t,bottom_left_latitude),	0, 0,
   "BOTTOM_RIGHT_LONGITUDE",	"%8.3f",	"%f",		T_FLOAT,	offsetof(gpds_tgeom_t,bottom_right_longitude),	0, 0,
   "BOTTOM_RIGHT_LATITUDE",	"%8.3f",	"%f",		T_FLOAT,	offsetof(gpds_tgeom_t,bottom_right_latitude),	0, 0,
   NULL,			NULL,		NULL,		0,		0,						0, 0
};

static pdstable_t pt_gvgeo[] = {
   "FEATURE_TYPE",		"\"%-20s\"",	"\"%.20s\"",	T_STRING,	offsetof(gpds_gvgeo_t,feature_type),		0, 0,
   "SEARCH_FEATURE_NAME",	"\"%-40s\"",	"\"%.40s\"",	T_STRING,	offsetof(gpds_gvgeo_t,feature_name),		0, 0,
   "DIACRITIC_FEATURE_NAME",	"\"%-40s\"",	"\"%.40s\"",	T_STRING,	offsetof(gpds_gvgeo_t,feature_diac),		0, 0,
   "FEATURE_STATUS_TYPE",	"\"%-15s\"",	"\"%.15s\"",	T_STRING,	offsetof(gpds_gvgeo_t,iau_status),		0, 0,

   "TOP_LEFT_LONGITUDE",	"%6.1f",	"%f",		T_FLOAT,	offsetof(gpds_gvgeo_t,top_left.longitude),	0, 0,
   "TOP_LEFT_LATITUDE",		"%6.1f",	"%f",		T_FLOAT,	offsetof(gpds_gvgeo_t,top_left.latitude),	0, 0,
   "TOP_LEFT_IMAGE_COORD_X",	"%6d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,top_left.image_coord_x),	0, 0,
   "TOP_LEFT_IMAGE_COORD_Y",	"%6d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,top_left.image_coord_y),	0, 0,
   "TOP_LEFT_TILE_COORD_X",	"%4d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,top_left.tile_coord_x),	0, 0,
   "TOP_LEFT_TILE_COORD_Y",	"%4d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,top_left.tile_coord_y),	0, 0,
   "TOP_LEFT_PIXEL_COORD_X",	"%4d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,top_left.pixel_coord_x),	0, 0,
   "TOP_LEFT_PIXEL_COORD_Y",	"%4d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,top_left.pixel_coord_y),	0, 0,

   "TOP_RIGHT_LONGITUDE",	"%6.1f",	"%f",		T_FLOAT,	offsetof(gpds_gvgeo_t,top_right.longitude),	0, 0,
   "TOP_RIGHT_LATITUDE",	"%6.1f",	"%f",		T_FLOAT,	offsetof(gpds_gvgeo_t,top_right.latitude),	0, 0,
   "TOP_RIGHT_IMAGE_COORD_X",	"%6d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,top_right.image_coord_x),	0, 0,
   "TOP_RIGHT_IMAGE_COORD_Y",	"%6d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,top_right.image_coord_y),	0, 0,
   "TOP_RIGHT_TILE_COORD_X",	"%4d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,top_right.tile_coord_x),	0, 0,
   "TOP_RIGHT_TILE_COORD_Y",	"%4d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,top_right.tile_coord_y),	0, 0,
   "TOP_RIGHT_PIXEL_COORD_X",	"%4d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,top_right.pixel_coord_x),	0, 0,
   "TOP_RIGHT_PIXEL_COORD_Y",	"%4d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,top_right.pixel_coord_y),	0, 0,

   "BOTTOM_LEFT_LONGITUDE",	"%6.1f",	"%f",		T_FLOAT,	offsetof(gpds_gvgeo_t,bottom_left.longitude),	   0, 0,
   "BOTTOM_LEFT_LATITUDE",	"%6.1f",	"%f",		T_FLOAT,	offsetof(gpds_gvgeo_t,bottom_left.latitude),	   0, 0,
   "BOTTOM_LEFT_IMAGE_COORD_X",	"%6d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,bottom_left.image_coord_x),  0, 0,
   "BOTTOM_LEFT_IMAGE_COORD_Y",	"%6d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,bottom_left.image_coord_y),  0, 0,
   "BOTTOM_LEFT_TILE_COORD_X",	"%4d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,bottom_left.tile_coord_x),   0, 0,
   "BOTTOM_LEFT_TILE_COORD_Y",	"%4d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,bottom_left.tile_coord_y),   0, 0,
   "BOTTOM_LEFT_PIXEL_COORD_X",	"%4d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,bottom_left.pixel_coord_x),  0, 0,
   "BOTTOM_LEFT_PIXEL_COORD_Y",	"%4d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,bottom_left.pixel_coord_y),  0, 0,

   "BOTTOM_RIGHT_LONGITUDE",	"%6.1f",	"%f",		T_FLOAT,	offsetof(gpds_gvgeo_t,bottom_right.longitude),	   0, 0,
   "BOTTOM_RIGHT_LATITUDE",	"%6.1f",	"%f",		T_FLOAT,	offsetof(gpds_gvgeo_t,bottom_right.latitude),	   0, 0,
   "BOTTOM_RIGHT_IMAGE_COORD_X","%6d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,bottom_right.image_coord_x), 0, 0,
   "BOTTOM_RIGHT_IMAGE_COORD_Y","%6d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,bottom_right.image_coord_y), 0, 0,
   "BOTTOM_RIGHT_TILE_COORD_X",	"%4d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,bottom_right.tile_coord_x),  0, 0,
   "BOTTOM_RIGHT_TILE_COORD_Y",	"%4d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,bottom_right.tile_coord_y),  0, 0,
   "BOTTOM_RIGHT_PIXEL_COORD_X","%4d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,bottom_right.pixel_coord_x), 0, 0,
   "BOTTOM_RIGHT_PIXEL_COORD_Y","%4d",		"%d",		T_INT,		offsetof(gpds_gvgeo_t,bottom_right.pixel_coord_y), 0, 0,

   NULL,			NULL,		NULL,		0,		0,						   0, 0
};
#endif /* GVPDSTAB_H */
