/*---------------------------------------------------------------*/ 
/*   Copyright (c) 1988 Regents of the University of California  */
/*                 All Rights Reserved                           */
/*---------------------------------------------------------------*/

#include "timec.h"

/*-- make_time_str -------------------------------------------
  Creates a string containing a plain text (human readable) 
  representation of a Cline time binary value. The string
  can be created in a variety of formats. The function
  returns a pointer to a text string which contains the
  formatted output. This text string is reused with each call.

  Entry Requirements:
    btime: The number of seconds from the beginning of the reference
	   year.
    format: The format code for the desired output.

  Return Value:
    A pointer to a string containing the given time in the desired
    format. The string is filled with "Invalid time format" if
    the given format code is unknown.

  Development History:
     Begun: 3/28/90 - Todd King

   Version:
       %W%    (UCLA/IGPP)   %G%
--------------------------------------------------------------*/
char *
make_time_str (double btime, int format)
{
  int yr, mon, day;
  int hour, min;
  int dsec, msec;
  float sec;
  static char timestr[60];

  if(btime >= 0.0 && btime < 10000000000.0) {  
     yrmonday(btime, &yr, &mon, &day, &hour, &min, &sec);

     switch(format) {
       case AMERDATE:
	 sprintf(timestr, "%02d/%02d/%4d %02d:%02d:%06.3f",
	     mon, day, yr, hour, min, sec);
	 break;
       case EURODATE:
	 sprintf(timestr, "%02d.%02d.%04d %02d:%02d:%06.3f",
	     day, mon, yr, hour, min, sec);
	 break;
       case ABBRAMER:
	 sprintf(timestr, "%3s %2d, %4d %02d:%02d:%06.3f",
	     Time_short_month_name[mon-1], day, yr, hour, min, sec);
	 break;
       case ABBREURO:
	 sprintf(timestr, "%02d %3s %4d %02d:%02d:%06.3f",
	     day, Time_short_month_name[mon-1], yr, hour, min, sec);
	 break;
       case LONGAMER:
	 sprintf(timestr, "%3s %2d, %4d %02d:%02d:%06.3f",
	     Time_long_month_name[mon-1], day, yr, hour, min, sec);
	 break;
       case LONGEURO:
	 sprintf(timestr, "%4d %3s %02d %02d:%02d:%06.3f",
	     yr, Time_long_month_name[mon-1], day, hour, min, sec);
	 break;
       case NUMERICAL:
	 sprintf(timestr, "%4d.%02d %02d:%02d:%06.3f",
	     yr, doy(yr, mon, day), hour, min, sec);
	 break;
       case DAYNUMBER:
	 sprintf(timestr, "%4d %03d %02d:%02d:%06.3f",
	    yr, doy(yr, mon, day), hour, min, sec);
	 break;
       case JAPANDATE:
	 sprintf(timestr, "%4d.%02d.%02d %02d:%02d:%06.3f",
	     yr, mon, day, hour, min, sec);
	 break;
       case NIPPONDATE:
	 sprintf(timestr, "%4d.%02d.%02d %02d:%02d:%06.3f",
	     yr, day, mon, hour, min, sec);
	 break;
       case HIGHLOW:
         dsec = sec;  /* Just the integer seconds */
         msec = (sec - dsec) * 1000; /* Just the millisec */
	 sprintf(timestr, "%4d %02d %02d %02d %02d %02d %03d",
	     yr, mon, day, hour, min, dsec, msec);
	 break;
       case ISEEDATE:
	 sprintf(timestr, "%4d %03d %3s %02d %02d:%02d:%06.3f",
	     yr, doy(yr, mon, day), 
             Time_short_month_name[mon-1], day, hour, min, sec);
	 break;
       case DFS_STYLE:
	 sprintf(timestr, "%4d-%3s-%02d %02d:%02d:%06.3f",
	     yr, Time_short_month_name[mon-1], day, hour, min, sec);
	 break;
       case ABBRDFS_STYLE:
	 sprintf(timestr, "%04d/%02d/%02d %02d:%02d:%06.3f",
	     yr, mon, day, hour, min, sec);
	 break;
       case PDS_STYLE:
	 sprintf(timestr, "%04d-%02d-%02dT%02d:%02d:%06.3f%c",
	     yr, mon, day, hour, min, sec, 'Z');
	 break;
       case ISO:
	 sprintf(timestr, "%04d%02d%02dT%02d%02d%06.3f%c",
	     yr, mon, day, hour, min, sec, 'Z');
	 break;
       case BINARY:
         sprintf(timestr, "%.3lf", btime);
         break;

     }
   } else {
       sprintf(timestr, "Invalid time value");
   }
   return(timestr);
}

