/*---------------------------------------------------------------*/ 
/*   Copyright (c) 1988 Regents of the University of California  */
/*                 All Rights Reserved                           */
/*---------------------------------------------------------------*/

#include "timec.h"
#include <string.h>

/*-- parse_time_str -----------------------------------------------
   Parses a time string and returns a binary time for
   what it specifies. A time string can be specified in a number
   of formats. The specific format a string is in ican be any of the
   supported formats (see the switch/case for more details).
   Any missing fields in the time string are set to default values
   which are the earliest possible value. Fields may be omitted
   only from the end of the string.

  Entry Requirements:
    tstr: A string containing a time specifications.
    format: The format code for the time specification in tstr.

  Return Value:
    The number of seconds from the beginning of the reference year.

   Development History:
      Begun: 3/28/90 - Todd King

   Version:
       %W%    (UCLA/IGPP)   %G%
--------------------------------------------------------------------*/
double 
parse_time_str (char tstr[], int format)
{
   double sectime();
   char *strrchr();

   int c;
   int day_of_year;
   char month_name[TIME_MAX_STR];
   int yr, mon, day, hour, min;
   float sec;
   int dsec, msec;
   double btime;
   char tmpstr[TIME_MAX_STR + 1];
   char *ptr;

   yr = REFYR;
   mon = 1;
   day = 1;
   hour = 0;
   min = 0;
   sec = 0.0;


   switch(format) {
       case AMERDATE:
	 sscanf(tstr, "%d/%d/%d %d:%d:%f",
	     &mon, &day, &yr, &hour, &min, &sec);
	 break;
       case EURODATE:
	 sscanf(tstr, "%d.%d.%d %d:%d:%f",
	     &day, &mon, &yr, &hour, &min, &sec);
	 break;
       case ABBRAMER:
	 sscanf(tstr, "%s %d, %d %d:%d:%f",
	     month_name, &day, &yr, &hour, &min, &sec);
	 mon = month_num(month_name);
	 break;
       case ABBREURO:
	 sscanf(tstr, "%d %s %d %d:%d:%f",
	     &day, month_name, &yr, &hour, &min, &sec);
	 mon = month_num(month_name);
	 break;
       case LONGAMER:
	 sscanf(tstr, "%s %d, %d %d:%d:%f",
	     month_name, &day, &yr, &hour, &min, &sec);
	 mon = month_num(month_name);
	 break;
       case LONGEURO:
	 sscanf(tstr, "%d %s %d %d:%d:%f",
	     &yr, month_name, &day, &hour, &min, &sec);
	 mon = month_num(month_name);
	 break;
       case NUMERICAL:
	 sscanf(tstr, "%d.%d %d:%d:%f",
	     &yr, &day_of_year, &hour, &min, &sec);
	 monthday(yr, day_of_year, &mon, &day);
	 break;
       case DAYNUMBER:
	 sscanf(tstr, "%d %d %d:%d:%f",
	    &yr, &day_of_year, &hour, &min, &sec);
	 monthday(yr, day_of_year, &mon, &day);
	 break;
       case JAPANDATE:
	 sscanf(tstr, "%d.%d.%d %d:%d:%f",
	     &yr, &mon, &day, &hour, &min, &sec);
	 break;
       case NIPPONDATE:
	 sscanf(tstr, "%d.%d.%d %d:%d:%f",
	     &yr, &day, &mon, &hour, &min, &sec);
	 break;
       case HIGHLOW:
	 sscanf(tstr, "%d %d %d %d %d %d %d",
	     &yr, &mon, &day, &hour, &min, &dsec, &msec);
         sec = dsec + (msec * 1000.0);
	 break;
       case ISEEDATE:
	 sscanf(tstr, "%d %d %s %d %d:%d:%f",
	     &yr, &day_of_year, month_name, &day, &hour, &min, &sec);
         mon = month_num(month_name);
	 break;
       case DFS_STYLE:
	 /* We must play with the time string all dashes with
	    a space. The reason is that scanf() seems unable
	    to distinguish the second dash as
	    a deliminator in this particular case. I've tried
	    character classes and other ways of dealing with it
	    with no success.                                      */
	 strncpy(tmpstr, tstr, TIME_MAX_STR);
	 tmpstr[TIME_MAX_STR] = '\0';
         while((ptr = strchr(tmpstr, '-')) != NULL) {
	    *ptr = ' ';
         }
	 sscanf(tmpstr, "%d %s %d %d:%d:%f",
	     &yr, month_name, &day, &hour, &min, &sec);
	 mon = month_num(month_name);
	 break;
       case ABBRDFS_STYLE:
	 sscanf(tstr, "%d/%d/%d %d:%d:%f",
	     &yr, &mon, &day, &hour, &min, &sec);
	 break;
       case PDS_STYLE:
	 sscanf(tstr, "%4d-%2d-%2dT%02d:%02d:%f%c",
	     &yr, &mon, &day, &hour, &min, &sec, &c);
	 break;
       case ISO:
	 sscanf(tstr, "%c%4d%2d%2dT%02d%02d%f%c",
	     &yr, &mon, &day, &hour, &min, &sec, &c);
	 break;
       case BINARY:	/* This returns early */
	 sscanf(tstr, "%lf", &btime);
	 return(btime);
       default: /* Unknown paradigm */
	  break;
    }

    btime = sectime(yr, mon, day, hour, min, sec);

    return(btime);
}

