/*---------------------------------------------------------------------
      Copyright(c) 1993 Regents of the University of California
      All rights reserved.
 
   Redistribution and use in source and binary forms are permitted
   provided that the above copyright notice and this paragraph are
   duplicated in all such forms and that any documentation,
   advertising materials, and other materials related to such
   distribution and use acknowledge that the software was developed
   by the University of California, Los Angeles.  The name of the
   University may not be used to endorse or promote products derived
   from this software without specific prior written permission.
   THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
   IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
   WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
-----------------------------------------------------------------------*/
/*------------------------------------------------------------------------
  Utility to extract information from a IGPP Flatfile format to any one of
  the available formats.

  Development History:
     Begun: 1/4/93 - Kirk Ketefian

  Version:
     @(#)ffextract.c	1.16        (UCLA/IGPP)     7/19/94
-------------------------------------------------------------------------*/

#include <stdio.h>
#include "ffextract.h"
#include "BC.h"               /* for data conversion */
#include "timec.h"            /* for PDS_STYLE */
#include <string.h>
#include <memory.h>           /* for memcpy() */

#ifdef VMS
#define unlink(filename) delete(filename)
#endif

FLATFILE *ffptr;              /* Flatfile pointer    */
FF_FULL_DESC *newdesc;        /* New descriptor for modified time fields */

char *EXTRACT_directives[] = {"ASCII", "TEXT", "HP", "IBM",
                              "MSB", "PC", "SUN", "VAX", ""};

enum {EXTRACT_ASCII = 0, EXTRACT_TEXT, EXTRACT_HP, EXTRACT_IBM,
      EXTRACT_MSB, EXTRACT_PC, EXTRACT_SUN, EXTRACT_VAX};

char *TIME_style_list[] = {"AMERDATE", "EURODATE", "ABBRAMER", "ABBREURO",
                          "LONGAMER", "LONGEURO", "NUMERICAL", "DAYNUMBER",
                          "JAPANDATE", "NIPPONDATE", "HIGHLOW", "ISEEDATE",
                          "DFS_STYLE", "ABBRDFS_STYLE", "PDS_STYLE", "ISO",
                          "BINARY", "GLL_SCLK", ""};


#define TIME_KEY             "TIME="
#define DEFAULT_TIMESTYLE    "PDS_STYLE"             /* default values */
#define DEFAULT_TS_VALUE      PDS_STYLE
#define DEFAULT_TIME_WIDTH    24
int     Time       =          DEFAULT_TS_VALUE;
int     Time_width =          DEFAULT_TIME_WIDTH;

/*-------------------------------------------------------------------------
  main()
--------------------------------------------------------------------------*/
int 
main (int argc, char *argv[])
{
   FILE *dat_fptr,
        *desc_fptr,
        *hed_fptr,
        *abs_fptr;
   char arg1[MAX_TEMP_LENGTH],
        arg2[MAX_TEMP_LENGTH],
        arg3[MAX_TEMP_LENGTH],
        in_ffile[MAX_TEMP_LENGTH],
        ASC_fname[MAX_TEMP_LENGTH],
        DAT_fname[MAX_TEMP_LENGTH],
        DES_fname[MAX_TEMP_LENGTH],
        HED_fname[MAX_TEMP_LENGTH],
        ABS_fname[MAX_TEMP_LENGTH],
        time_style[MAX_TEMP_LENGTH],
	Open_style[10],
        *ptr;
   int Is_ascii = 0,
       time_arg = 0,
       Format,
       i, j;

   /* scan command line for "TIME=____" option */
   for(i=1; i < argc; i++) {
      strcpy(time_style, argv[i]);
      time_style[strlen(TIME_KEY)] = '\0';
      scaseup(time_style);
      if (strcmp(time_style, "TIME=") == 0) {
         ptr = &argv[i][strlen(TIME_KEY)];
         strcpy(time_style, ptr);
         /* rearrange args to get rid of "TIME=xxxxx" argument */
         for(j=i; j < argc-1; j++) {
            argv[j] = argv[j+1];
         }
         argv[j] = NULL;      /* get rid of last argument */
         argc--;
         time_arg = 1;
         break;
      }
   }

   /* set time_style - report error if unkown timestyle requested */
   if(time_arg == 1) {
      if(set_time_style(time_style) != DB_SUCCESS) {
         printf("Timestyle \"%s\" not recognized.  Defaulting to \"%s\".\n",
                                              time_style, DEFAULT_TIMESTYLE);
      }
   }

   if (argc < 2) {
      do_help(stdout);
      exit(1);
   }

   if (argc < 3) {
      strcpy(arg1, "ASCII");       /* format         */
      strcpy(arg2, argv[1]);       /* input flatfile */
   } else {  /* argc >= 3 */
      strcpy(arg1, argv[1]);       /* format          */
      scaseup(arg1);
      strcpy(arg2, argv[2]);       /* input flatfile  */
      if (argc > 3) {
         strcpy(arg3, argv[3]);    /* output flatfile */
      } else {  /*  no output file specified. Output ascii to screen */
         Is_ascii = 1;
      }
   }

   Format = kw_match(arg1, EXTRACT_directives);
   if (Format < 0) {  /* No match */
      fprintf(stderr, "Unknown Format: %s\n", arg1);
      exit(1);
   }
   
   switch(Format) {
      case EXTRACT_TEXT:
      case EXTRACT_ASCII:
         Is_ascii = 1;
         break;
   }

   PickVarFormat(CURRENT_PLATFORM);

   /* try to open the given flatfile */
   strcpy(in_ffile, arg2);
   if ((ffptr = ffopen(in_ffile, "r")) == NULL) {
      remove_extension(in_ffile);
      if ((ffptr = ffopen(in_ffile, "r")) == NULL) {
         fprintf(stderr, "Unable to open flatfile: %s\n", arg2);
         exit(1);
      }
   }

   /* initialize a new descriptor info struct if requested output */
   /* is ascii and requested time style is not Binary.            */
   if ((Is_ascii) && (Time != BINARY)) {
      init_newdesc();
   }

   /* we know that <format> and <Input Flatfile> are provided */
   if (argc > 3) {                      /* <Outfile> provided */
      strcpy(DAT_fname, arg3);
      trimspc(DAT_fname);
      if (Is_ascii) {
         strcat(DAT_fname, ".ASC");
         strcpy(Open_style, "w");
      } else {
         strcat(DAT_fname, ".DAT");
         strcpy(Open_style, "wb");
      }
      if (exists_file(DAT_fname)) {
         fprintf(stderr, "'%s' aborted!\n", argv[0]);
         exit(1);
      }
      if ((dat_fptr = fopen(DAT_fname, Open_style)) == NULL) {
         fprintf(stderr, "Unable to write to output file: %s\n", DAT_fname);
         ffclose(ffptr);
         exit(1);
      }
   } else {          /* no <Outfile> provided, list to screen */
      list_flatfile("", ffptr);
      exit(0);
   }

   switch(Format) {
      case EXTRACT_ASCII:
      case EXTRACT_TEXT:
         PickVarFormat(CURRENT_PLATFORM);
         extract_flatfile(ASCII_FRMT, dat_fptr, ffptr);
         break;
/*    case EXTRACT_HP:
         PickVarFormat("hp");
         extract_flatfile(BINARY_FRMT, dat_fptr, ffptr);
         break;
      case EXTRACT_MSB:
         PickVarFormat("msb");
         extract_flatfile(BINARY_FRMT, dat_fptr, ffptr);
         break;
*/    case EXTRACT_IBM:
         PickVarFormat("ibm");
         extract_flatfile(BINARY_FRMT, dat_fptr, ffptr);
         break;
      case EXTRACT_PC:
         PickVarFormat("pc");
         extract_flatfile(BINARY_FRMT, dat_fptr, ffptr);
         break;
      case EXTRACT_SUN:
         PickVarFormat("sun");
         extract_flatfile(BINARY_FRMT, dat_fptr, ffptr);
         break;
      case EXTRACT_VAX:
         PickVarFormat("vax");
         extract_flatfile(BINARY_FRMT, dat_fptr, ffptr);
         break;
      default:            /* will never get here */
         break;
   }
   PickVarFormat(CURRENT_PLATFORM);
   fclose(dat_fptr);

   if (argc > 3) {       /* output file specified */
      /* write out Descriptor file */
      strcpy(DES_fname, arg3);
      trimspc(DES_fname);
      strcat(DES_fname, ".DES");
      if (exists_file(DES_fname)) {
         fprintf(stderr, "Descriptor file '%s' not written.\n", DES_fname);
      } else if ((desc_fptr = fopen(DES_fname, "w")) == NULL) {
         fprintf(stderr, "Unable to write to descriptor file: %s\n", DES_fname);
      } else {
         write_desc(desc_fptr, Is_ascii);
         fclose(desc_fptr);
      }
   }
   if ((argc > 3) && !(Is_ascii)) {  /* output specified & format NOT ascii */
      /* write out Header file */
      strcpy(HED_fname, arg3);
      trimspc(HED_fname);
      strcat(HED_fname, ".HED");
      if (exists_file(HED_fname)) {
         fprintf(stderr, "Header file '%s' not written.\n", HED_fname);
      } else if ((hed_fptr = fopen(HED_fname, "w")) == NULL) {
         fprintf(stderr, "Unable to write to header file: %s\n", HED_fname);
      } else {
         write_hed(hed_fptr);
         fclose(hed_fptr);
      }
      /* write out Abstract file */
      strcpy(ABS_fname, arg3);
      trimspc(ABS_fname);
      strcat(ABS_fname, ".ABS");
      if (exists_file(ABS_fname)) {
         fprintf(stderr, "Abstract file '%s' not written.\n", ABS_fname);
      } else if ((abs_fptr = fopen(ABS_fname, "w")) == NULL) {
         fprintf(stderr, "Unable to write to abstract file: %s\n", ABS_fname);
      } else {
         write_abs(abs_fptr);
         fclose(abs_fptr);
      }
   }
      
   ffclose(ffptr);
}


/*-------------------------------------------------------------------------
  do_help() - prints usage information.
--------------------------------------------------------------------------*/
void
do_help(FILE *fptr)
{
   fprintf(fptr, "FFEXTRACT is a command-line utility that converts UCLA/IGPP Flatfiles from the\n");
   fprintf(fptr, "native SUN binary format (MSB/IEEE) to any one of the formats specified below.\n\n");
   fprintf(fptr, "  Usage:  ffextract [<Format>] <Input Flatfile> [<Outfile>] [time=<TimeStyle>]\n");
   fprintf(fptr, "          ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n");
   fprintf(fptr, "where brackets \"[  ]\" indicate optional arguments.\n");
   fprintf(fptr, "<Format> is the format of the output file and may be one of the following:\n\n");
   fprintf(fptr, "        ibm   =  IBM 370 binary format\n");
   fprintf(fptr, "        vax   =  VAX/VMS binary format\n");
/* fprintf(fptr, "        hp    =  HP 1000 binary format\n");                 */
/* fprintf(fptr, "        msb   =  Intel PC using Microsoft binary format\n");*/
   fprintf(fptr, "        pc    =  Intel PC using IEEE binary format\n");
   fprintf(fptr, "        sun   =  SUN binary format (MSB/IEEE) -- no conversion\n");
   fprintf(fptr, "        ascii =  ASCII format (text file)\n\n");
   fprintf(fptr, "<Input Flatfile> is the name of the UCLA/IGPP Flatfile to be opened.\n\n");
   fprintf(fptr, "<Outfile> is the optional name of the Flatfile or ASCII file (depending on\n");
   fprintf(fptr, "the format requested) in which the output is to be placed.  If the <Format>\n");
   fprintf(fptr, "and <Outfile> arguments are not specified, the output will be sent to the\n");
   fprintf(fptr, "terminal screen in ASCII format.\n\n");
   fprintf(fptr, "<TimeStyle> specifies the ascii output format of the time fields.  The default\n");
   fprintf(fptr, "style is PDS_STYLE (See help file for a listing of possible time styles).\n");
}


/*-------------------------------------------------------------------------
  find_record() - looks for the record with the 1st column name "refname".
     It starts from record 0 and returns FF_SUCCESS if it encounters the
     desired record, otherwise returns FF_FAILURE.
--------------------------------------------------------------------------*/
int 
find_record (FLATFILE *ffptr, char refname[])  
{
   char temp[MAX_TEMP_LENGTH];
 
   ffmove_to_rec(ffptr, 0);
  
   while (ffread_rec(ffptr) == FF_SUCCESS) {
      Sun_To_Format(ffptr);
      strcpy(temp, ffget_col_str(ffptr, 0));  /* first column (field) */
      trimspc(temp);
      if(strcmp(temp, refname) == 0)
         return(FF_SUCCESS);
   }  
   return(FF_FAILURE); 
}


/*-------------------------------------------------------------------------
  list_recs() - prints all the records in the flatfile in a tabular format
     starting with start_no.
--------------------------------------------------------------------------*/
int 
list_recs (FLATFILE *ffptr, unsigned long start_no)
{
   unsigned long i;
   char temp[MAX_TEMP_LENGTH];
   FF_FULL_DESC *desc;
   char tmpbuff[64];
   unsigned long lcount = 0;
   int increment = 1, more_flag = 0;
   int begin = 0, end  = 0;
   unsigned long top = start_no + 1,
                 bottom = start_no + MAX_LINES;

   if (ffptr == NULL) return (FF_FAILURE);

   PrintDES(ffptr, stdout);
   printf("<Return> to continue...");
   fgets(tmpbuff, sizeof(tmpbuff), stdin);
   if((tmpbuff[0] == 'q') || (tmpbuff[0] == 'Q')) {
      return(FF_SUCCESS);
   }

   if (bottom < ffptr->nrecs)
      do_calc_and_header(ffptr, begin, &end, top, bottom);
   else  do_calc_and_header(ffptr, begin, &end, top, ffptr->nrecs);

   ffmove_to_rec(ffptr, start_no);

   while (ffread_rec(ffptr) == FF_SUCCESS) {
      Sun_To_Format(ffptr);
      if (begin > 0)
         printf("|~");
      for (i=begin; i < end; i++) {
         strcpy(temp, ffget_col_str(ffptr, i));
         trimspc(temp);

         desc = ffget_desc(ffptr, i);
         switch(desc->type) {
            case FF_INT:
               printf("| %8.8s ", temp);   /* 10 */
               break;
            case FF_FLOAT:
               printf("| %12.12s ", temp);
               break;
            case FF_DOUBLE:
               printf("| %20.20s ", temp);
               break;
            case FF_ALPHA:
               pad_front_with_spaces(temp, desc->width - strlen(temp));
               printf("| %s ", temp);
               break;
            case FF_TIME:
               sprintf(temp, "%s", make_time_str(ffget_col_val(ffptr,i),Time));
               pad_front_with_spaces(temp, Time_width - strlen(temp)); 
               printf("| %s ", temp);
               break;
         }
      }

      if (end >= ffptr->ncols)
         printf("|\n");
      else printf("|~|\n");

      lcount += increment;
      if ((lcount >= MAX_LINES) || (ffptr->cur_recnum == ffptr->nrecs)) {
         if (more_flag == 1) break; /* break out of while loop */ 
         printf("<Return> to advance, <R>ight, <L>eft, <H> for help...");
         fgets(tmpbuff, sizeof(tmpbuff), stdin);
         switch(tmpbuff[0]) {
            case 'q':
            case 'Q':
               return(FF_SUCCESS);
               break;
            case 'm':
            case 'M':
               increment = 0;
               more_flag = 1;
               break;
            case 'l':
            case 'L':
               if (begin > 0) {
                  begin--;
               }
               ffmove_to_rec(ffptr, top - 1);
               break;
            case 'r':
            case 'R':
               if (end < ffptr->ncols) {
                  begin++;
               }
               ffmove_to_rec(ffptr, top - 1);
               break;
            case 'd':
            case 'D':
               PrintDES(ffptr, stdout); 
               printf("<Return> to continue..."); 
               fgets(tmpbuff, sizeof(tmpbuff), stdin);
               if((tmpbuff[0] == 'q') || (tmpbuff[0] == 'Q')) {
                  return(FF_SUCCESS);
               }
               ffmove_to_rec(ffptr, top - 1);
               break;
            case 'b':
            case 'B':
               if (top >= MAX_LINES + 1) {
                  ffmove_to_rec(ffptr, top-MAX_LINES-1);
                  top = top - MAX_LINES;
               } else {
                  ffmove_to_rec(ffptr, 0);
                  top = 1;
               }
               break;
            case '?':
            case 'h':
            case 'H':
               Print_list_help(stdout);
               ffmove_to_rec(ffptr, top - 1);
               break;
            default:     /* when advancing forward */
               top = ffptr->cur_recnum + 1;
         } /* end switch */

         lcount = 0;

         bottom = top + MAX_LINES - 1;
         if (more_flag == 0) {
            if (ffptr->cur_recnum < ffptr->nrecs) {
               if (bottom < ffptr->nrecs)
                  do_calc_and_header(ffptr, begin, &end, top, bottom);
               else  do_calc_and_header(ffptr, begin, &end, top, ffptr->nrecs);
            }
         }
      } /* end if */
   } /* end while */

   printf("\n");
   return(FF_SUCCESS);
}


/*-------------------------------------------------------------------------
  Print_list_help() - prints the help information on the list command.
--------------------------------------------------------------------------*/
void
Print_list_help(FILE *fptr)
{
   char tmpbuff[64];
   int i;

   for (i=0; i < MAX_LINES; i++)
      fprintf(fptr, "\n");
   fprintf(fptr, "                            HELP INFORMATION\n");
   for (i=3; i < MAX_WIDTH; i++)
      fprintf(fptr, "-");
   fprintf(fptr, "\n");
   for (i=3; i < MAX_WIDTH; i++)
      fprintf(fptr, "-");
   fprintf(fptr, "\n");

   fprintf(fptr, "  ?,H    :  Display this help information.\n");
   fprintf(fptr, "   Q     :  Exit listing.\n");
   fprintf(fptr, "   M     :  Non-stop display of the remaining records.\n");
   fprintf(fptr, "   L     :  Shift the window to the left.\n");
   fprintf(fptr, "   R     :  Shift the window to the right.\n");
   fprintf(fptr, "   D     :  Display the Record Descriptor Information.\n");
   fprintf(fptr, "   B     :  Go back to previous screen.\n");
   fprintf(fptr, "<Return> :  Advance to next screen.\n");

   for (i=3; i < MAX_WIDTH; i++)
      fprintf(fptr, "-");
   fprintf(fptr, "\n");
   for (i=3; i < MAX_WIDTH; i++)
      fprintf(fptr, "-");
   fprintf(fptr, "\n\n\n\n\n\n\n\n\n\nPress <Return> to resume listing...");
   fgets(tmpbuff, sizeof(tmpbuff), stdin);
}


/*-------------------------------------------------------------------------
  do_calc_and_header() - calculates "end" and prints header.
--------------------------------------------------------------------------*/
int 
do_calc_and_header (FLATFILE *ffptr, int begin, int *end, unsigned long top_rec, unsigned long bottom_rec)
{
   FF_FULL_DESC *desc;
   int i, tag = 0, total = 1;
   char temp[MAX_WIDTH + 1], temp2[MAX_WIDTH + 1],
        out_line[MAX_WIDTH + 1];


   if (ffptr == NULL) return (FF_FAILURE);

   out_line[0] = '\0'; temp[0] = '\0';

   if (begin > 0)   /* account for the '|~' at the beginning */
      total += 2;

   for (i=begin; i < ffptr->ncols; i++) {
      desc = ffget_desc(ffptr, i);
      switch(desc->type) {
         case FF_INT:
            if (total + 11 < MAX_WIDTH) {    /* 13 */
               total += 11;
               sprintf(temp, "| %8s ", desc->shortname);
            } else tag = 1;
            break;
         case FF_FLOAT:
            if (total + 15 < MAX_WIDTH) {
               total += 15;
               sprintf(temp, "| %12s ", desc->shortname);
            } else tag = 1;
            break;
         case FF_DOUBLE:
            if (total + 23 < MAX_WIDTH) {
               total += 23;
               sprintf(temp, "| %20s ", desc->shortname);
            } else tag = 1;
            break;
         case FF_ALPHA:
            if (total + desc->width + 3 < MAX_WIDTH) {
               total += desc->width + 3;
               sprintf(temp2, "%s", desc->shortname);
               pad_front_with_spaces(temp2, desc->width - strlen(temp2));
               sprintf(temp, "| %s ", temp2);
            } else tag = 1;
            break;
         case FF_TIME:
            if (total + Time_width+3 < MAX_WIDTH) {
               total += Time_width+3;
#ifdef REMOVED
               if(Time == BINARY) {   /* keep same descriptor name if ctime */
                  sprintf(temp2, "%s", desc->shortname);
               } else {   /* not BINARY */
                  sprintf(temp2, "%s", TIME_style_list[Time]);
               }
#endif
               sprintf(temp2, "%s", desc->shortname);
               pad_front_with_spaces(temp2, Time_width - strlen(temp2));
               sprintf(temp, "| %s ", temp2);
            } else tag = 1;
            break;
      }

      if (tag == 1) {
         tag = 0;
         break;
      }
      strncat(out_line, temp, MAX_WIDTH);
   }

   *end = i;

   if (*end < ffptr->ncols)
      total += 2;

   /* print range of records displayed */
   printf("Viewing Rec# %ld to %ld", top_rec, bottom_rec);
   if (top_rec == 1)
      printf("  *** BEGINNING OF FLATFILE ***");
   else if (bottom_rec == ffptr->nrecs)
      printf("  *** END OF FLATFILE ***");
   printf("\n");

   /* print top line ------- */
   for (i=0; i < total; i++)
      printf("-");
   printf("\n");

   /* print header information */
   if (begin > 0)
      printf("|~");
   printf("%s|", out_line);
   if (*end >= ffptr->ncols)
      printf("\n");
   else printf("~|\n");

   /* print bottom line ------- */
   for (i=0; i < total; i++)
      printf("-");
   printf("\n");
}



/*-------------------------------------------------------------------------
  pad_front_with_spaces() - pads the given string with 'count' number of
     spaces in the front.
--------------------------------------------------------------------------*/
int 
pad_front_with_spaces (char *string, int count)
{
   int i;
   char tempstring[MAX_TEMP_LENGTH];

   strcpy(tempstring, string);
   for(i=0; i < count; i++)
      string[i] = ' ';
   string[i] = '\0';
   strcat(string, tempstring);
}


/*-------------------------------------------------------------------------
  list_flatfile()
--------------------------------------------------------------------------*/
int 
list_flatfile (char refname[], FLATFILE *ffptr)
{
   char temp[MAX_TEMP_LENGTH];
   unsigned long rec_number = 0;


   if (ffptr == NULL)
      return(DB_FAILURE);


   if (refname[0] == '\0') {
      ffmove_to_rec(ffptr, 0);
      if (list_recs(ffptr, 0) == FF_FAILURE) {
         return(DB_FAILURE);
      }
      return(DB_SUCCESS);
   } else {               /* refname[0] != '\0' */
      if (refname[0] == '#') {
         strcpy(temp, refname);
         remove_num_sign(temp);
         rec_number = (unsigned long)atol(temp);
         if ((rec_number <= 0) || (rec_number > ffptr->nrecs))
            return(DB_FAILURE);
         if (list_recs(ffptr, rec_number - 1) == FF_FAILURE)
            return(DB_FAILURE);
      } else {          /* search for 'refname' in first field */
         if (find_record(ffptr, refname) == FF_FAILURE)
            return(DB_FAILURE);
         if (list_recs(ffptr, ffptr->cur_recnum - 1) == FF_FAILURE) {
                              /* - 1 in order to read the needed record */
            return(DB_FAILURE);
         }
      }
      return(DB_SUCCESS);
   }
}


/*-----------------------------------------------------------------------
  remove_num_sign() - gets rid of '#' at the beginning of the given
     string.  If no '#' exists at the beginning of string, does nothing.
------------------------------------------------------------------------*/
int 
remove_num_sign (char *string)
{
   char temp[MAX_TEMP_LENGTH];
   int i;

   if (string[0] == '#') {
      strcpy(temp, string);
      for(i=1; temp[i] != '\0'; i++)
         string[i-1] = temp[i];
      string[i-1] = '\0';
      return(DB_SUCCESS);
   }
}


/*-----------------------------------------------------------------------
  remove_extension() - if string/filename has an extension, such as .DAT
     this function removes it from the string.
------------------------------------------------------------------------*/
int 
remove_extension (char *filename)
{
   int i = 0;
 
   if (filename[0] == '\0')       /* if filename is the empty string */
      return(DB_SUCCESS);
 
   for (i = strlen(filename) - 1; i >= 0; i--) {
      if (filename[i] == '.') {
         filename[i] = '\0';
         return(DB_SUCCESS);
      }
   }
   return(DB_SUCCESS);
}


/*-----------------------------------------------------------------------
  PrintDES() - displays the contents of the descriptor (sub)file in a
     flatfile.
  Written by  - Gilbert Hyatt
  Modified by - Kirk Ketefian  (7/21/92)
------------------------------------------------------------------------*/
void
PrintDES(FLATFILE *ffptr, FILE *outfile)
{
   FF_FULL_DESC  *descptr;
   char tmpbuff[64];
   unsigned long lcount = 3,
                 increment = 1;

 
   fprintf(outfile, "Record Descriptor Information\n");
   fprintf(outfile, "-----------------------------------------------------------------------------\n");
   fprintf(outfile, "|L|       Long Column Name         |  Name  |     Units      |T| len |  sort|\n");
   fprintf(outfile, "-----------------------------------------------------------------------------\n");
 
   if (Time != BINARY) {
      descptr = newdesc;
   } else descptr = ffptr->descdb;

   for (/*descptr = ffptr->descdb*/; descptr != NULL; descptr = descptr->next) {
      fprintf(outfile, "|  ");
      fprintf(outfile, "%-32s|",      descptr->longname);
      fprintf(outfile, "%-8s|",       descptr->shortname);
      fprintf(outfile, "%-16s|",      descptr->units);
      fprintf(outfile, "%c|",         descptr->type);
      fprintf(outfile, " %-3d |",     descptr->width);
      fprintf(outfile, "  %-2d  |",   descptr->sort);

      lcount += increment;
      if (lcount > MAX_LINES + 2) {
         fflush(stdout);
         fgets(tmpbuff, sizeof(tmpbuff), stdin);
         if (tmpbuff[0] == 'm')
            increment = 0;
         lcount = 0;
      } else fprintf(outfile, "\n");
   }
 
   fprintf(outfile, "-----------------------------------------------------------------------------\n\n");
}


/*-----------------------------------------------------------------------
  init_newdesc() - changes the BINARY time to the specified format in the
     descriptor information of the flatfile pointer.
------------------------------------------------------------------------*/
int 
init_newdesc (void)
{
   FF_FULL_DESC *desc1, *desc2;
   char temp[MAX_TEMP_LENGTH];
   int i;

   newdesc = NULL;
   for (desc1 = ffptr->descdb; desc1 != NULL; desc1 = desc1->next) {
      desc2 = (FF_FULL_DESC *)malloc(sizeof(FF_FULL_DESC));
      if (newdesc == NULL) newdesc = desc2;  /* will be done only once */
      memcpy((char *)desc2, (char *)desc1, sizeof(*desc1));
      desc2 = desc2->next;
   }

   for (desc1 = newdesc; desc1 != NULL; desc1 = desc1->next) {
      if(desc1->type == FF_TIME) {
#ifdef REMOVED
         sprintf(temp, "Time in %s format", TIME_style_list[Time]);
         strncpy(desc1->longname, temp, FF_LNAME);
         strncpy(desc1->shortname, TIME_style_list[Time], FF_SNAME);
         strncpy(desc1->units, "time", FF_UNITS);
#endif
         desc1->type = FF_ALPHA;
         desc1->width = (unsigned int)time_style_width(Time);
      }
   }
}


/*-----------------------------------------------------------------------
  Sun_To_Format() -  Used in conjunction with PickVarFormat().
------------------------------------------------------------------------*/
int 
Sun_To_Format (FLATFILE *ffptr)
{
  FF_FULL_DESC  *descptr;
  char          *ptr;
 
  for (descptr = ffptr->descdb; descptr != NULL; descptr = descptr->next) {
    ptr = ffptr->rec_buff + descptr->offset;
    switch(descptr->type) {
      case FF_INT:
        BCIEEEToInteger((int *)ptr);
        break;
      case FF_FLOAT:
        BCIEEEToFloat((float *)ptr);
        break;
      case FF_DOUBLE:
      case FF_TIME:
        BCIEEEToDouble((double *)ptr);
        break;
      case FF_ALPHA:
        BCASCIIToChar(ptr, descptr->width);
        break;
      default:
        break;
    }
  }
}


/*-----------------------------------------------------------------------
  Format_To_Sun() -  Used in conjunction with PickVarFormat().
------------------------------------------------------------------------*/
int 
Format_To_Sun (FLATFILE *ffptr)
{
  FF_FULL_DESC  *descptr;
  char          *ptr;

  for (descptr = ffptr->descdb; descptr != NULL; descptr = descptr->next) {
    ptr = ffptr->rec_buff + descptr->offset;
    switch(descptr->type) {
      case FF_INT:
        BCIntegerToIEEE((int *)ptr);
        break;
      case FF_FLOAT:
        BCFloatToIEEE((float *)ptr);
        break;
      case FF_DOUBLE:
      case FF_TIME:
        BCDoubleToIEEE((double *)ptr);
        break;
      case FF_ALPHA:
        BCCharToASCII(ptr, descptr->width);
        break;
      default:
        break;
    }
  }
}
 

/*-----------------------------------------------------------------------
  PickVarFormat() - selects the right conversion functions for the
     desired format.

  Written by  - Gilbert Hyatt
------------------------------------------------------------------------*/
int 
PickVarFormat (char *formatstr)
{
  switch(*formatstr) {
    /*  Convert to IBM 370 Format  */
    case 'I':
    case 'i': BCIBM370Format();
              break;
    /*  Convert to VAX/VMS Format  */
    case 'V':
    case 'v': BCVAXFormat();
              break;
    /*  Convert to HP 1000 Format  */
    case 'H':
    case 'h': BCHP1000Format();
              break;
    /*  Convert to Intel PC using Microsoft binary Format */
    case 'M':
    case 'm': BCMSBFormat();
              break;
    /*  Convert to Intel PC using IEEE Format  */
    case 'P':
    case 'p': BCPCFormat();
              break;
    /*  Convert to Sun Format      */
    case 'S':
    case 's': BCSunFormat();
              break;
    /*  Non of the above           */
    default:
      return(0);
      break;
  }
  return(1);
}


/*-----------------------------------------------------------------------
  extract_flatfile()
------------------------------------------------------------------------*/
int
extract_flatfile(int style, FILE *fptr, FLATFILE *ffptr)
{
   if ((fptr == NULL) || (ffptr == NULL))
      return(DB_FAILURE);

   if (style == ASCII_FRMT)
      ascii_extract(ffptr, 0, fptr);
   else binary_extract(ffptr, 0, fptr);     /* style == BINARY_FRMT */

   return(DB_SUCCESS);
}


/*-----------------------------------------------------------------------
  ascii_extract()
------------------------------------------------------------------------*/
int
ascii_extract(FLATFILE *ffptr, unsigned long start_no, FILE *fptr)
{
   unsigned long i;
   char temp[MAX_TEMP_LENGTH];
   FF_FULL_DESC *desc;
   int begin = 0,
        end  = ffptr->ncols;
 
 
   if (ffptr == NULL) return (FF_FAILURE);
   if (fptr == NULL) return (FF_FAILURE);
 
   PrintHeader(ffptr, fptr);

   ffmove_to_rec(ffptr, start_no);
 
   while (ffread_rec(ffptr) == FF_SUCCESS) {
      Sun_To_Format(ffptr);
      for (i=begin; i < end; i++) {
         strcpy(temp, ffget_col_str(ffptr, i));
         trimspc(temp);
 
         desc = ffget_desc(ffptr, i);
         switch(desc->type) {
            case FF_INT:
               fprintf(fptr, "%8s", temp);   /* 10 */
               break;
            case FF_FLOAT:
               fprintf(fptr, "%12s", temp);
               break;
            case FF_DOUBLE:
               fprintf(fptr, "%20s", temp);
               break;
            case FF_ALPHA:
               pad_front_with_spaces(temp, desc->width - strlen(temp));
               fprintf(fptr, "%s", temp);
               break;
            case FF_TIME:
               fprintf(fptr,"%s", make_time_str(ffget_col_val(ffptr,i), Time));
               break;
         }
         if (i < end - 1)
            fprintf(fptr, " ");
      }
      fprintf(fptr, "\n");
   }
   return(FF_SUCCESS);
}

/*-----------------------------------------------------------------------
  PrintHeader() - prints the header information of the ascii output file.
    Used by ascii_extract().
------------------------------------------------------------------------*/
void
PrintHeader(FLATFILE  *ffptr, FILE *outfile)
{
  FF_FULL_DESC  *descptr;
  int           index, width;
  char          charbuffer[(FF_MAX_FIELD_WIDTH +1)],
                temp[MAX_TEMP_LENGTH];
 
  for (descptr = ffptr->descdb; descptr != NULL; descptr = descptr->next)
    switch(descptr->type) {
      case FF_INT:
      case FF_FLOAT:
         fprintf(outfile, "%10s ", descptr->shortname);
         break;
      case FF_DOUBLE:
         fprintf(outfile, "%20s ", descptr->shortname);
         break;
      case FF_TIME:
#ifdef REMOVED
         if(Time == BINARY) {   /* keep same descriptor name if binary */
            sprintf(temp, "%s", descptr->shortname);
         } else {   /* not BINARY */
            sprintf(temp, "%s", TIME_style_list[Time]);
         }
#endif
         sprintf(temp, "%s", descptr->shortname);
         pad_front_with_spaces(temp, Time_width - strlen(temp));
         fprintf(outfile, "%s ", temp);
         break;
      case FF_ALPHA:
         width = (descptr->width > FF_SNAME) ? descptr->width: FF_SNAME;
         sprintf(charbuffer, "%%%ds ", width);
         fprintf(outfile, charbuffer, descptr->shortname);
         break;
      }  /* end of switch */
  fprintf(outfile, "\n");
}



/*-----------------------------------------------------------------------
  binary_extract()
------------------------------------------------------------------------*/
int
binary_extract( FLATFILE *ffptr, unsigned long start_no, FILE *fptr)
{
   unsigned long i;
   char temp[MAX_TEMP_LENGTH];
   FF_FULL_DESC *desc;
   int begin = 0,
        end  = ffptr->ncols;

 
   if (ffptr == NULL) return (FF_FAILURE);
   if (fptr == NULL) return (FF_FAILURE);
 
   ffmove_to_rec(ffptr, start_no);
 
   while (ffread_rec(ffptr) == FF_SUCCESS) {
      Sun_To_Format(ffptr);
      if (fwrite(ffptr->rec_buff, ffptr->reclen, 1, fptr) != 1)
         return(FF_FAILURE);
   }    
   return(FF_SUCCESS);
}


/*-----------------------------------------------------------------------
  write_desc()
------------------------------------------------------------------------*/
int
write_desc( FILE *desc_fptr, int Is_ascii)
{
   FF_FULL_DESC *dptr;
   char line[MAX_TEMP_LENGTH + 1];
   int i;


   if ((desc_fptr == NULL) || (ffptr->desc == NULL))
      return(DB_FAILURE);

   if ((Is_ascii) && (Time != BINARY)) {
      dptr = newdesc;
   } else {
      dptr = ffptr->descdb;
   }

   while(dptr != NULL) {
      fprintf(desc_fptr, "%-32s%-8s%-16s%c%-3d%2d\n",
               dptr->longname,
               dptr->shortname,
               dptr->units,
               dptr->type,
               dptr->width,
               dptr->sort   );
      dptr = dptr->next;
   }

   return(DB_SUCCESS);
}


/*-----------------------------------------------------------------------
  write_hed()
------------------------------------------------------------------------*/
int
write_hed( FILE *hed_fptr)
{
   char line[MAX_TEMP_LENGTH + 1];

   if ((hed_fptr == NULL) || (ffptr->hed == NULL))
      return(DB_FAILURE);

   fseek(ffptr->hed, 0, 0);
   while(fgets(line, MAX_TEMP_LENGTH, ffptr->hed) != NULL) {
      fprintf(hed_fptr, "%s", line);
   }

   return(DB_SUCCESS);
}


/*-----------------------------------------------------------------------
  write_abs()
------------------------------------------------------------------------*/
int
write_abs(FILE *abs_fptr)
{
   char line[MAX_TEMP_LENGTH + 1];

   if ((abs_fptr == NULL) || (ffptr->abs == NULL))
      return(DB_FAILURE);

   fseek(ffptr->abs, 0, 0);
   while(fgets(line, MAX_TEMP_LENGTH, ffptr->abs) != NULL) {
      fprintf(abs_fptr, "%s", line);
   }

   return(DB_SUCCESS);
}


/*--------------------------------------------------------------------
  exists_file() - checks to see if there is already a file with
     the name "filename", and if so asks if it should be replaced.
     Returns 1 if the file exists, 0 if doesn't exist or if it is
     OK to replace it.
---------------------------------------------------------------------*/
int 
exists_file (char *filename)
{
   FILE *temp_fptr;
   char tmpbuff[64];
 
 
   if ((temp_fptr = fopen(filename, "r")) == NULL)
      return(0);
   fclose(temp_fptr);
   fprintf(stdout,"The file '%s' already exists. Would you like to replace it? ",
                                                                       filename); 
   fgets(tmpbuff, sizeof(tmpbuff), stdin);
   if ((tmpbuff[0] == 'y') || (tmpbuff[0] == 'Y'))
      return(0);
   return(1);
}



/*--------------------------------------------------------------------
  set_time_style() - tries to set the desired timestyle to time_style.
    If time_style is not recognized, it defaults to DEFAULT_TS_VALUE.
---------------------------------------------------------------------*/
int 
set_time_style (char *time_style)
{
   trimspc(time_style);
   scaseup(time_style);

   Time = kw_match(time_style, TIME_style_list);

   if(Time < 0) {
      Time = DEFAULT_TS_VALUE;
      Time_width = time_style_width(Time);
      return(DB_FAILURE);
   }

   Time_width = time_style_width(Time);

   return(DB_SUCCESS);
}
