/*---------------------------------------------------------------*/
/*   Copyright (c) 1989 Regents of the University of California  */
/*                 All Rights Reserved                           */
/*---------------------------------------------------------------*/
/*
      %W%  (UCLA, IGPP) %G%
*/
/*-- ffmove_bin ----------------------------------------------------
  Sets the current record to be the one which contains the field with
  the specified value. A binary search is performed so the specified
  field MUST be sorted in ascending order.
  If no field exists with the specified value the record with the
  nearest value is selected. This call positions the 
  file pointer at the beginning of the record. To extract the 
  contents of the record you must make a call to ffread_rec()
  or something similar.

   Entry Requirements:
    ffptr: A flat file pointer (as returned from ffopen() )
    cidx: A number indicating the column index to search on.
    value: The vlaue to search for.
 
  Return Value:
    FF_SUCCESS: If it was possible to re-position the current record.
    FF_FAILURE: If the flatfile pointer is invalid or the move was
                impossible.

  Development History:
    Begun: 11/1/89 - Todd King
---------------------------------------------------------------------*/
#include "ffio.h"

int 
ffmove_bin (FLATFILE *ffptr, unsigned long cidx, double value)
{
  double curval;
  double lastval;
  unsigned long cur_rec;
  unsigned long bottom, last_bottom;
  unsigned long top, last_top;

  if(ffptr == NULL) return(FF_FAILURE);

/* Check special cases of search value not within the flatfile
   at all or it is the first or last value.
*/
  cur_rec = 0;
  if(ffmove_to_rec(ffptr, cur_rec) == FF_FAILURE) return(FF_FAILURE);
  ffread_rec(ffptr);
  curval = ffget_col_val(ffptr, cidx);
  if(curval >= value) {
    ffmove_to_rec(ffptr, cur_rec);   /* reposition to begining of record */
    return(FF_SUCCESS);
  }
  lastval = curval;

  cur_rec = ffptr->nrecs - 1;	/* moving is base zero indexed */
  if(ffmove_to_rec(ffptr, cur_rec) == FF_FAILURE) return(FF_FAILURE);
  ffread_rec(ffptr);
  curval = ffget_col_val(ffptr, cidx);
  if(curval > lastval) {	/* Its probably monotonically increasing */
    if(curval < value) return(FF_SUCCESS);
    if(curval == value) {
      ffmove_to_rec(ffptr, cur_rec);   /* reposition to begining of record */
      return(FF_SUCCESS);
    }
  }

  /* Try to home in on the right record */
  bottom = 0;
  top = cur_rec;
  for(;;) {
    cur_rec = bottom + (top - bottom) / 2;
    if(ffmove_to_rec(ffptr, cur_rec) == FF_FAILURE) return(FF_FAILURE);
    ffread_rec(ffptr);
    curval = ffget_col_val(ffptr, cidx);
    if(curval == value) {  /* found it */
      ffmove_to_rec(ffptr, cur_rec);   /* reposition to begining of record */
      break;
    }
    last_bottom = bottom;
    last_top = top;
    if(curval < value) {
      bottom = cur_rec;
    } else {
      top = cur_rec;
    }
    if(last_bottom == bottom && last_top == top) { break; } /* Exhausted */
  }

  return(FF_SUCCESS);
}

