/*---------------------------------------------------------------*/
/*   Copyright (c) 1993 Regents of the University of California  */
/*                 All Rights Reserved                           */
/*---------------------------------------------------------------*/
/*
      %W%  (UCLA, IGPP) %G%
*/
/*-- ffopen_select ------------------------------------------
  Re-opens all or selective portions of a flatfile for reading
  or writing given a flat file pointer. Returns the pointer 
  passed if successful or NULL is unable to open the desired 
  component. The counter portions of the flatfile structure 
  are not altered unless the data component of the flatfile 
  is opened.

  A component flatfile has each portion of a flatfile in 
  a seperate file. The descriptor portion is the definition of the
  structure that all flatfiles refered to in the data file must also
  have.

  Entry Requirements:
     ffptr: A pointer to a flatfile (as returned from ffopen()).
     option: either "r" for reading, "w" for writing, "a" for
             append, or "u" for update (insert or delete). 
             If the option is "a" the flatfile must already
             exist and only the data portion and abstract portion
             of the flatfile may be appended to.
     compon: A composite flag which is specified with the
	     parameters of FF_DAT, FF_DES,which
	     specifies that the ".FFD"and  ".FFH"
	     file is to be opened. The special paramter FF_ALL
	     is defined to be all the files. An example composite
	     flag is:

		FF_DAT|FF_DES

	      which specifies to open the ".DAT" and ".DES" files.

  Return Value:
      A flatfile pointer, the pointer is null if the open failed.

  Development History: (ffopen_select.c)
     Begun: 9/05/89 - Todd King

  Development History: (lffopen_select.c)
  Begun: 03/15/93 - Gilbert Hyatt
  Edited:09/20/93 - Gilbert Hyatt 
     added support for upper and lower case extensions.
-------------------------------------------------------------*/
#include <sys/types.h>
#include <sys/stat.h>	/* For stat() */
#include <unistd.h>	/* For access() */
#include "ffio.h"

/* Lower Flatfile Parameters */
#define BUFFERSIZE      4096
#define FFH_REC_LEN     72

/* Format table for .FFD descriptor file */
#define FFD_LNAME_START 24
#define FFD_LNAME_END   48
#define FFD_SNAME_START 4
#define FFD_SNAME_END   11
#define FFD_UNITS_START 14
#define FFD_UNITS_END   22
#define FFD_TYPE_START  50
#define FFD_TYPE_END    50
#define FFD_WIDTH_START 56
#define FFD_WIDTH_END   58	


FLATFILE *
lffopen_select (FLATFILE *ffptr, char option[], unsigned int compon)
{
  char tmpname[MAX_FILENAME];
  char tmpopt[10];
  unsigned long cnt;
  char buffer[FF_DESCWIDTH];
  char tmpbuffer[FF_DESCWIDTH];
  char *ptr;
  FF_DESC *desc_ptr;
  struct stat stat;
  int i;

  /* added by Gilbert */
  char         longname[FF_LNAME + 1], shortname[FF_SNAME + 1], units[FF_UNITS + 1];
  char         type, ttype[3];
  unsigned int width;
  char         DESline[100], temp_str[9];
  int          col_count;


  /* Check validity of options */
  if(ffptr == NULL) return(NULL);

  if(strlen(option) > 1) { return(NULL); }

  switch(option[0]) {
     case 'w':
	ffptr->open_style = FF_OPEN_WRITE;
	break;
     case 'r':
	ffptr->open_style = FF_OPEN_READ;
	break;
     case 'a':
	ffptr->open_style = FF_OPEN_APPEND;
	break;
     case 'u':
	ffptr->open_style = FF_OPEN_UPDATE;
	break;
     default:
	return(NULL);
  }  /* end of switch */

  
  /* Only subport reading, appending, and updating of existing Flowfiles */
  if(ffptr->open_style == FF_OPEN_WRITE) return(NULL);

  /* Preset */
  /********  OPEN DESCRIPTOR FILE ******/
  if ((compon & FF_DES) && !(ffptr->open_status & FF_DES) ) {
    if (ffptr->open_style == FF_OPEN_APPEND || ffptr->open_style == FF_OPEN_UPDATE) 
         strcpy(tmpopt, "r");
    else strcpy(tmpopt, option);

    /* See if file exists */
    strcpy(tmpname, ffptr->fname);
    strcat(tmpname, ".ffh");
    if((ffptr->desc = fopen(tmpname, tmpopt)) == NULL) {
      strcpy(tmpname, ffptr->fname);
      strcat(tmpname, ".FFH");
      if((ffptr->desc = fopen(tmpname, tmpopt)) == NULL) { return(NULL); }
    }  /* end if */


    if (! (ffptr->open_status & FF_OPENED_BEFORE)) {/*Check if loaded before.If so don't reload*/
      if (tmpopt[0] == 'r') { /* Load internal data base with values */

	/*** Get information about Descriptor File ***/
	/* Skip first three lines: DATA,CDATE */
	for (i=0;i<3;i++) fread((void *)DESline,FFH_REC_LEN,1,ffptr->desc);
	if (ferror(ffptr->desc)) return(NULL);

	/* Read in NCOLS */
	fread((void *)DESline,FFH_REC_LEN,1,ffptr->desc);
	if(ferror(ffptr->desc)) return(NULL);
	segment(DESline,temp_str,7,12);
	col_count = atoi(temp_str);

	/* Check for .FFH format error */
	if (col_count==0) return(NULL);

	/* Skip next three lines: NROWS,OPSYS and # NAME */
	for (i=0;i<3;i++)  fread((void *)DESline,FFH_REC_LEN,1,ffptr->desc);
	if (ferror(ffptr->desc)) return(NULL);

	/* FILL STRINGS LONGER THAN THOSE IN .FFD WITH BLANKS */
	memset(units,    0, (FF_UNITS + 1));
	memset(longname, 0, (FF_LNAME +1));

	/*** Get Descriptor Elements ***/
	for(i=0;i<col_count;i++) {
	  fread((void *)DESline,FFH_REC_LEN,1,ffptr->desc);
	  if(ferror(ffptr->desc)) return(NULL);
	  
	  /* LONGNAME */
	  segment(DESline,longname,FFD_LNAME_START,FFD_LNAME_END);
	  /* SHORTNAME */
	  segment(DESline,shortname,FFD_SNAME_START,FFD_SNAME_END);
	  /* UNITS */
	  segment(DESline,units,FFD_UNITS_START,FFD_UNITS_END);
	  /* TYPE */
	  segment(DESline,ttype,FFD_TYPE_START,FFD_TYPE_END);
          type = ttype[0];
	  if (!valid_type(type)) return(NULL);
	  /* WIDTH */
	  width = (type == 'T' || type == 'D')?8:4;
	  if(i == 0 && type == 'D') type = 'T';
	  ffmakedesc(ffptr, longname, shortname, units, type, width, 0);
	}  /* end of for */
      }  /* end of if */

      ffptr->open_status |= FF_OPENED_BEFORE;
      rewind(ffptr->desc);
    }  /* end of if */

    ffptr->open_status |= FF_DES;
  }  /* end of if */



  /******* OPEN DATA FILE ********/
  if ((compon & FF_DAT) && !(ffptr->open_status & FF_DAT) ) {
     if(ffptr->open_style == FF_OPEN_UPDATE) strcpy(tmpopt, "r+");
     else strcpy(tmpopt, option);
     strcat(tmpopt, "b");
     
     strcpy(tmpname, ffptr->fname);
     strcat(tmpname, ".ffd");
     if((ffptr->data = fopen(tmpname, tmpopt)) == NULL) {
       strcpy(tmpname, ffptr->fname);
       strcat(tmpname, ".FFD");
       if((ffptr->data = fopen(tmpname, tmpopt)) == NULL) { return(NULL); }
    }  /* end of if */

    if(ffptr->reclen > 0) {	/* Determine number of records */
      fstat(fileno(ffptr->data), &stat);
      ffptr->nrecs = stat.st_size / ffptr->reclen;
    }
    ffptr->cur_recnum = 0;

    if(ffptr->open_style == FF_OPEN_APPEND) ffmove_to_rec(ffptr, ffptr->nrecs);
  }

  ffptr->type        =  FF_TYPE_LOWER;
  ffptr->open_status |= FF_DAT;
  return(ffptr);
}


int 
segment (char *instr, char *outstr, int start, int end)
{
  int i,j;
  
  for(i=start,j=0;i<=end;i++,j++) outstr[j] = instr[i];
}  /* end of segment() */
 
 
int 
insert (char *instr, char *ins, int start, int end)
{
  int i,j;
  
  for(i=start,j=0;i<=end;i++,j++) instr[i] = ins[j];
}  /* end of insert() */
 
int 
valid_type (int c)
{
  switch (c)
  {
    case 'I':
    case 'R':
    case 'D':
    case 'T': return 1;
    default:  return 0;
  }
}
