/*---------------------------------------------------------------*/
/*   Copyright (c) 1992 Regents of the University of California  */
/*                 All Rights Reserved                           */
/*---------------------------------------------------------------*/
/*-----------------------------------------------------------------
  Binary conversion functions for an IBM 370.
  The IBM 370 stores all numeric values in MSB first format.

  Development History:
    Begun: 03/20/92 - Todd King
       Based on code written by Gordon Maclean (UCLA).
    Modified: 7/28/93 - Kirk Ketefian
       Rewrote all Float and Double functions using code adapted from:
	  Val I. Garger - vig@eagle.cnsf.cornell.edu
	  Technology Integration Group
	  CNSF, Cornell University
    Modified: 8/24/93 - Kirk Ketefian
       Added code to enable proper execution on the VAX.
    Modified: 5/26/94 - Kirk : added MSDOS support

  Version:
      %W%     (UCLA/IGPP)    %G%
-----------------------------------------------------------------*/
#include "BC.h"

/* =================================================================== *
 * function:      IBMFloatToIEEE(float ibm)                            *
 * ------------------------------------------------------------------- *
 * Development History:                                                *
 *   Begun: 07/27/93 - Kirk Ketefian                                   *
 *                                                                     *
 *   Adapted from code by:                                             *
 *       Val I. Garger - vig@eagle.cnsf.cornell.edu                    *
 *       Technology Integration Group                                  *
 *       CNSF, Cornell University                                      *
 * ------------------------------------------------------------------- *
 *                                                                     *
 * Format (bits, left to right):            |    Exponent bias:        *
 *              sign   exponent   mantissa  |                          *
 *  IBM           1      7           24     |     64 hex               *
 *  IEEE          1      8           23     |     127                  *
 *                                          |                          *
 * Usage notes:                                                        *
 * 1. Data is converted "inplace".                                     *
 * 2. IBM values that do not conform to IEEE standard are converted to *
 *    either infinite IEEE values (positive or negative)  or to zero.  *
 * 3. Non-normalized with zero exponent values are kept intact.        *
 * 4. Conversion does not incur the loss of mantissa accuracy.         *
 * =================================================================== */

#define   expon  0x7F000000
#define   Sign   0x80000000
#define   tissa  0x00FFFFFF
#define   etis   0x007FFFFF
#define   norm   0x00F00000
 
void 
IBM370FloatToIEEE (int *ibm)
{
   int long ibs, ibe, ibt, it, k;
   union { int long i; float r; } u;
 
#if defined(vax) || defined(MSDOS)
   BCReverseByteOrder(ibm, sizeof(float));
#endif
   ibs = *ibm & Sign;
   ibe = *ibm & expon;
   ibt = *ibm & tissa;
 
   if  (ibt == 0) {
      ibe = 0;
   } else {
      if ((ibe != 0) && (ibt & norm) == 0) {
	 u.i = *ibm;
	 u.r = u.r + 0e0;
	 ibe = u.i & expon;
	 ibt = u.i & tissa;
      }
 
      /* mantissa */
      it = ibt << 8;
      for (k = 0; (k < 5) && (it >= 0); k++ )
	 it = it << 1;
 
      if ( k < 4 ) {
	 ibt = (it >> 8) & etis;
	 ibe = (ibe >> 22) - 256 + 127 - k - 1;
	 if (ibe < 0) {
	    ibe = ibt = 0;
	 }
	 if (ibe >= 255) {
	    ibe = 255;
	    ibt = 0;
	 }
	 ibe = ibe << 23;
      }
   }
 
   *ibm = ibs | ibe | ibt;
#if defined(vax) || defined(MSDOS)
   BCReverseByteOrder(ibm, sizeof(float));
#endif
}


/* =================================================================== *
 * function:      IBM370IEEEToFloat(float ieee)                        *
 * ------------------------------------------------------------------- *
 * Development History:                                                *
 *   Begun: 07/28/93 - Kirk Ketefian                                   *
 *                                                                     *
 *   Adapted from code by:                                             *
 *       Val I. Garger - vig@eagle.cnsf.cornell.edu                    *
 *       Technology Integration Group                                  *
 *       CNSF, Cornell University                                      *
 * ------------------------------------------------------------------- *
 *                                                                     *
 * Format (bits, left to right):            |    Exponent bias:        *
 *              sign   exponent   mantissa  |                          *
 *  IBM           1      7           24     |     64 hex               *
 *  IEEE          1      8           23     |     127                  *
 *                                          |                          *
 * Usage notes:                                                        *
 * 1. Data is converted "inplace".                                     *
 * 2. Infinite IEEE values are converted to the largest IBM values     *
 *    which are x'7FFFFFFF' and x'FFFFFFFF' for positive and negative  *
 *    respectively.                                                    *
 * 3. Like infinite values, NaN (Not a Number) values are converted to *
 *    the largest values.                                              *
 * 4. Precision in the mantissa could be lost by rounding off the      *
 *    least significant bits.         0 <= |error| <= 0.24E-6          *
 *    (From 0 to 3 least significant bits out of 24 mantissa bits      *
 *    could be rounded.)                                               *
 * =================================================================== */
 
#define last 0x000000ff
#define impl 0x00800000
#define sign 0x80000000
#define tiss 0x007fffff

void 
IBM370IEEEToFloat (int *ieee)
{
   int long k, ibs, ibe, ibt;
 
#if defined(vax) || defined(MSDOS)
   BCReverseByteOrder(ieee, sizeof(float));
#endif
   ibt = *ieee;
   ibs = *ieee & sign;
   ibe = (*ieee >> 23) & last;
 
   if (ibe != 0) {
      if (ibe == 255) {
	 ibe = 378;
	 ibt = tiss;
      }
      ibe = ibe - 127 + 256 + 1;
      k = ibe%4;
      ibe = ibe >> 2;
      if (k != 0) {
	 ibe = ibe + 1;
      }
      ibe = ibe << 24;
      ibt = (ibt & tiss) | impl;
      if (k != 0) {
	 ibt = (ibt + (1 << (3-k))) >> (4-k);
      }
   }
   *ieee = ibs | ibe | ibt;
#if defined(vax) || defined(MSDOS)
   BCReverseByteOrder(ieee, sizeof(float));
#endif
}


/* =================================================================== *
 * function:     IBM370DoubleToIEEE(double ibm)                        *
 * ------------------------------------------------------------------- *
 * Development History:                                                *
 *   Begun: 07/28/93 - Kirk Ketefian                                   *
 *                                                                     *
 *   Adapted from code by:                                             *
 *       Val I. Garger - vig@eagle.cnsf.cornell.edu                    *
 *       Technology Integration Group                                  *
 *       CNSF, Cornell University                                      *
 * ------------------------------------------------------------------- *
 *                                                                     *
 * Format (bits, left to right):            |    Exponent bias:        *
 *              sign   exponent   mantissa  |                          *
 *  IBM           1      7           56     |     64 hex               *
 *  IEEE          1     11           52     |     1023                 *
 *                                          |                          *
 * Usage notes:                                                        *
 * 1. Data is converted "inplace".                                     *
 * 2. Non-normalized with zero exponent values are kept intact.        *
 * 3. Precision in the mantissa could be lost by rounding off the      *
 *    least significant bits.     0 <= |error| <= .56E-16              *
 *    (From 0 to 3 least significant bits out of 56 mantissa bits      *
 *    could be rounded.)                                               *
 * =================================================================== */
 
#define  exp    0x7f000000
#define  sign   0x80000000
#define  mtiss  0x00FFFFFF
#define  nrm    0x00f00000
 
#define  look0  0x00800000
#define  look1  0x00400000
#define  look2  0x00200000
#define  look3  0x00100000
 
#define  take0  0x007fffff
#define  take1  0x003fffff
#define  take2  0x001fffff
#define  take3  0x000fffff

void 
IBM370DoubleToIEEE (int ibm[])
{
   int ibs, ibe, ibt1, ibt2, it, k, isht;
 
   union {
      struct {
	 int long i1;
	 int long i2;
      } sti;
      struct {
	 double r;
      } str;
   } un;
 
#if defined(vax) || defined(MSDOS)
   BCReverseByteOrder(&ibm[0], sizeof(float));
   BCReverseByteOrder(&ibm[1], sizeof(float));
#endif
   ibs  = ibm[0] & sign;
   ibe  = ibm[0] & exp;
   ibt1 = ibm[0] & mtiss;
   ibt2 = ibm[1];
 
   if ((ibt1 == 0) && (ibt2 == 0)) {
      ibe = 0;
   } else {
      if ((ibe != 0) && ((ibt1 & nrm) == 0)) {
	 un.sti.i1 = ibm[0];
	 un.sti.i2 = ibm[1];
	 un.str.r  = un.str.r + 0e0;
	 ibe  = un.sti.i1 & exp;
	 ibt1 = un.sti.i1 & mtiss;
	 ibt2 = un.sti.i2;
      }
 
      if ((ibt1 & look0) != 0) {
	 k = 3;
	 ibt1 = ibt1 & take0;
      } else if ((ibt1 & look1) != 0) {
	 k = 2;
	 ibt1 = ibt1 & take1;
      } else if ((ibt1 & look2) != 0) {
	 k = 1;
	 ibt1 = ibt1 & take2;
      } else if ((ibt1 & look3) != 0) {
	 k = 0;
	 ibt1 = ibt1 & take3;
      } else {
	 k = 4;
      }

      if (k < 4) {
	 if (k != 0) {    /* shift with rounding */
	    it   = (ibt2 & 65535) + (1 << (k-1));
	    ibt2 = ((ibt2 >> 16) & 65535) + (it >> 16);
	    ibt1 = ibt1 + (ibt2 >> 16);
	     /* rounded */
	    ibt2 = ((ibt2 & 65535) << (16-k)) | ((it & 65535) >> k);
	    it   = ibt1 << (32 - k);
	    ibt1 = ibt1 >> k;
	    ibt2 = ibt2 | it;
	 }
 
	 /*  exponent */
	 ibe = ( (ibe >> 22)  - 256 + 1023 + k - 4 ) << 20 ;
      }
   }
 
   /* done */
   ibm[0] = ibs | ibe | ibt1;
   ibm[1] = ibt2;
#if defined(vax) || defined(MSDOS)
   BCReverseByteOrder(&ibm[0], sizeof(float));
   BCReverseByteOrder(&ibm[1], sizeof(float));
#endif
}


/* =================================================================== *
 *  function:      IBM370IEEEToDouble(double ieee)                     *
 * ------------------------------------------------------------------- *
 * Development History:                                                *
 *   Begun: 07/28/93 - Kirk Ketefian                                   *
 *                                                                     *
 *   Adapted from code by:                                             *
 *       Val I. Garger - vig@eagle.cnsf.cornell.edu                    *
 *       Technology Integration Group                                  *
 *       CNSF, Cornell University                                      *
 * ------------------------------------------------------------------- *
 *                                                                     *
 * Format (bits, left to right):           |    Exponent bias:         *
 *             sign   exponent   mantissa  |                           *
 * IBM           1      7           56     |     64 hex                *
 * IEEE          1     11           52     |     1023                  *
 *                                         |                           *
 * Usage notes:                                                        *
 * 1. Data is converted "inplace".                                     *
 * 2. The infinite numbers and NaN (Not a Number) values are converted *
 *    to the largest IBM values.                                       *
 * 3. IEEE values that do not fit to IBM standard are converted to     *
 *    either the biggest IBM values (positive or negative) or to zero. *
 * 4. Conversion does not incur the loss of mantissa accuracy.         *
 * =================================================================== */
 
#define  Impl 0x00100000
#define  sign 0x80000000
#define  maxl 0x00FFFFFF
#define  expn 0x7FF00000
#define  Tiss 0x000FFFFF
#define  maxr 0xFFFFFFFF
 
void 
IBM370IEEEToDouble (int ieee[])
{
   int k, ibs, ibe, ibt1, ibt2, isht;
 
#if defined(vax) || defined(MSDOS)
   BCReverseByteOrder(&ieee[0], sizeof(float));
   BCReverseByteOrder(&ieee[1], sizeof(float));
#endif
   ibs  = ieee[0] & sign;
   ibe  = ieee[0] & expn;
   ibt1 = ieee[0] & Tiss;
   ibt2 = ieee[1];
 
   if (ibe != 0) {
      ibe = ibe >> 20;
      ibe = ibe - 1023 + 256 + 1;
 
      k = 0;
      if (ibe > 508) {
	 k = 2;
      }
      if (ibe < 0) {
	 k = 1;
      }
 
      switch (k) {
	 case 1:
	    ibe = ibt1 = ibt2 = 0;
	    break;
	 case 2:
	    ibe  = 127;
	    ibt1 = maxl;
	    ibt2 = maxr;
	    break;
	 default:
	    isht = ibe%4 - 1;
	    ibe = ibe >> 2;
	    if (isht != -1) {
	       ibe = ibe + 1;
	    } else {
	       isht = 3;
	    }
	    ibt1 = ibt1 | Impl;
	    if (isht != 0) {
	       ibt1 = (ibt1 << isht) |  (ibt2 >> 32-isht);
	       ibt2 = ibt2 << isht;
	    }
	    break;
      } /* end switch */
 
      ibe = ibe << 24;
   }

   ieee[0] = ibs | ibe | ibt1;
   ieee[1] = ibt2;
#if defined(vax) || defined(MSDOS)
   BCReverseByteOrder(&ieee[0], sizeof(float));
   BCReverseByteOrder(&ieee[1], sizeof(float));
#endif
}


/*-------------------------------------------------------------
   Convert ASCII characters to EBCDIC, in place.  This
   uses the map contained in the VMS routine LIB$TRA_ASC_EBC.
   Any ASCII characters which have no EBCDIC equivalents
   are converted to EBCDIC '?'.

   The length of the string is passed.

RETURN VALUE
  no value, except value of argument is changed.

AUTHOR/DATE
  Gordon Maclean  UCLA Inst. of Geophysics   May 91
  Modified : 03/30/92 - Todd King
	      Adapted to BC calling.
--------------------------------------------------------------------*/
void 
IBM370ASCIIToChar (unsigned char buffer[], int len)
{
  static char table[]={                                         /* octal */
       '\000','\001','\002','\003','\067','\055','\056','\057',  /* 000 */
       '\026','\005','\045','\013','\014','\015','\016','\017',  /* 010 */
       '\020','\021','\022','\023','\074','\075','\062','\046',  /* 020 */
       '\030','\031','\077','\047','\034','\035','\036','\037',  /* 030 */
       '\100','\117','\177','\173','\133','\154','\120','\175',  /* 040 */
       '\115','\135','\134','\116','\153','\140','\113','\141',  /* 050 */
       '\360','\361','\362','\363','\364','\365','\366','\367',  /* 060 */
       '\370','\371','\172','\136','\114','\176','\156','\157',  /* 070 */
       '\174','\301','\302','\303','\304','\305','\306','\307',  /* 100 */
       '\310','\311','\321','\322','\323','\324','\325','\326',  /* 110 */
       '\327','\330','\331','\342','\343','\344','\345','\346',  /* 120 */
       '\347','\350','\351','\112','\340','\132','\137','\155',  /* 130 */
       '\171','\201','\202','\203','\204','\205','\206','\207',  /* 140 */
       '\210','\211','\221','\222','\223','\224','\225','\226',  /* 150 */
       '\227','\230','\231','\242','\243','\244','\245','\246',  /* 160 */
       '\247','\250','\251','\300','\152','\320','\241','\007',  /* 170 */
       '\077','\077','\077','\077','\077','\077','\077','\077',
       '\077','\077','\077','\077','\077','\077','\077','\077',
       '\077','\077','\077','\077','\077','\077','\077','\077',
       '\077','\077','\077','\077','\077','\077','\077','\077',
       '\077','\077','\077','\077','\077','\077','\077','\077',
       '\077','\077','\077','\077','\077','\077','\077','\077',
       '\077','\077','\077','\077','\077','\077','\077','\077',
       '\077','\077','\077','\077','\077','\077','\077','\077',
       '\077','\077','\077','\077','\077','\077','\077','\077',
       '\077','\077','\077','\077','\077','\077','\077','\077',
       '\077','\077','\077','\077','\077','\077','\077','\077',
       '\077','\077','\077','\077','\077','\077','\077','\077',
       '\077','\077','\077','\077','\077','\077','\077','\077',
       '\077','\077','\077','\077','\077','\077','\077','\077',
       '\077','\077','\077','\077','\077','\077','\077','\077',
       '\077','\077','\077','\077','\077','\077','\077','\377'};

   int i;

   for (i=0; i < len; i++) {
      buffer[i] = table[buffer[i]];
   }

   return;
}

/*----------------------------------------------------------
   Convert EBCDIC characters to ASCII, in place.  This
   uses the map contained in the VMS routine LIB$TRA_EBC_ASC.
   Any EBCDIC characters which have no ASCII equivalents
   are converted to '\'.  Other strange translations are:

   EBCDIC               Ascii
   IBM logical not      ^
   cent sign            [
   !                    ]
   |                    !

   The length of the string is passed.

RETURN VALUE
  no value, except value of argument is changed.

AUTHOR/DATE
  Gordon Maclean  UCLA Inst. of Geophysics   Feb 89
  Modified : 03/30/92 - Todd King
	      Adapted to BC calling.
--------------------------------------------------------------------*/
void 
IBM370CharToASCII (unsigned char buffer[], int len)
{
   static char table[]={                                       /* octal */
     '\000','\001','\002','\003',  '\\','\011',  '\\','\177',  /* 000 */
       '\\',  '\\',  '\\','\013','\014','\015','\016','\017',  /* 010 */
     '\020','\021','\022','\023',  '\\',  '\\','\010',  '\\',  /* 020 */
     '\030','\031',  '\\',  '\\','\034','\035','\036','\037',  /* 030 */
       '\\',  '\\',  '\\',  '\\',  '\\','\012','\027','\033',  /* 040 */
       '\\',  '\\',  '\\',  '\\',  '\\','\005','\006','\007',  /* 050 */
       '\\',  '\\','\026',  '\\',  '\\',  '\\',  '\\','\004',  /* 060 */
       '\\',  '\\',  '\\',  '\\','\024','\025',  '\\','\032',  /* 070 */
	' ',  '\\',  '\\',  '\\',  '\\',  '\\',  '\\',  '\\',  /* 100 */
       '\\',  '\\',   '[',   '.',   '<',   '(',   '+',   '!',  /* 110 */
	'&',  '\\',  '\\',  '\\',  '\\',  '\\',  '\\',  '\\',  /* 120 */
       '\\',  '\\',   ']',   '$',   '*',   ')',   ';',   '^',  /* 130 */
	'-',   '/',  '\\',  '\\',  '\\',  '\\',  '\\',  '\\',  /* 140 */
       '\\',  '\\',   '|',   ',',   '%',   '_',   '>',   '?',  /* 150 */
       '\\',  '\\',  '\\',  '\\',  '\\',  '\\',  '\\',  '\\',  /* 160 */
       '\\',   '`',   ':',   '#',   '@',  '\'',   '=',  '\"',  /* 170 */
       '\\',   'a',   'b',   'c',   'd',   'e',   'f',   'g',  /* 200 */
	'h',   'i',  '\\',  '\\',  '\\',  '\\',  '\\',  '\\',  /* 210 */
       '\\',   'j',   'k',   'l',   'm',   'n',   'o',   'p',  /* 220 */
	'q',   'r',  '\\',  '\\',  '\\',  '\\',  '\\',  '\\',  /* 230 */
       '\\',   '~',   's',   't',   'u',   'v',   'w',   'x',  /* 240 */
	'y',   'z',  '\\',  '\\',  '\\',  '\\',  '\\',  '\\',  /* 250 */
       '\\',  '\\',  '\\',  '\\',  '\\',  '\\',  '\\',  '\\',  /* 260 */
       '\\',  '\\',  '\\',  '\\',  '\\',  '\\',  '\\',  '\\',  /* 270 */
	'{',   'A',   'B',   'C',   'D',   'E',   'F',   'G',  /* 300 */
	'H',   'I',  '\\',  '\\',  '\\',  '\\',  '\\',  '\\',  /* 310 */
	'}',   'J',   'K',   'L',   'M',   'N',   'O',   'P',  /* 320 */
	'Q',   'R',  '\\',  '\\',  '\\',  '\\',  '\\',  '\\',  /* 330 */
       '\\',  '\\',   'S',   'T',   'U',   'V',   'W',   'X',  /* 340 */
	'Y',   'Z',  '\\',  '\\',  '\\',  '\\',  '\\',  '\\',  /* 350 */
	'0',   '1',   '2',   '3',   '4',   '5',   '6',   '7',  /* 360 */
	'8',   '9',  '\\',  '\\',  '\\',  '\\',  '\\','\377'}; /* 370 */

   int i;

   for (i=0; i < len; i++) {
      buffer[i] = table[buffer[i]];
   }

   return;
}


void 
BCIBM370Format (void)
{
   BCFloatToIEEE = IBM370FloatToIEEE;
   BCIEEEToFloat = IBM370IEEEToFloat;

   BCDoubleToIEEE = IBM370DoubleToIEEE;
   BCIEEEToDouble = IBM370IEEEToDouble;

   BCGFloatToIEEE = BCDoNothing;
   BCIEEEToGFloat = BCDoNothing;

   BCShortToIEEE = BCDoNothing;
   BCIEEEToShort = BCDoNothing;

   BCIntegerToIEEE = BCDoNothing;
   BCIEEEToInteger = BCDoNothing;

   BCLongIntegerToIEEE = BCDoNothing;
   BCIEEEToLongInteger = BCDoNothing;

   BCCharToASCII = IBM370CharToASCII;
   BCASCIIToChar = IBM370ASCIIToChar;

   BCReverseByteOrder = BCGenericReverseByteOrder;
}
