/*---------------------------------------------------------------*/
/*   Copyright (c) 1988 Regents of the University of California  */
/*                 All Rights Reserved                           */
/*---------------------------------------------------------------*/
/*
      %W%  (UCLA, IGPP) %G% 
 */

/*-- MAKEDESC ---------------------------
 Makes a new descriptor in the data
 base at the next available location.
 Width can only be set for 'A' types
 For the other types system sizes are
 taken.
 Returns -1 if impossible to do what
 was requested, the current number of
 descriptors otherwise.

 Entry Requirements:
  LONGC: The long column name to use.
 SHORTC: The short column name to use.
  UNITS: The units label to use.
   TYPE: The data type of the column
         (i.e. A, D, I, R, T)
  WIDTH: The width of the column in
         bytes.
   SORT: The sort order of the column.

 Return values:
    0: All went well
   -1: Descriptors already written
       Too late to make any more
   -2: Maximum number of columns
   -3: Invalid type
   -4: Maximum record length exceeded

 Development History:
   Begun: 2/21/86 - Todd King
            Under GH06 support.
   Update: 6/9/87  - Marco Zagha
      (Check to make sure descriptors
       have not been written out)
            Under GG44 support.
   Update: 7/2/87 - Todd King
      Fixed bug, field width was not
      being properly set.
      Added more informative return
      values
            Under GH06 support.
  Edited:04/14/94 - Gilbert Hyatt
    Now uses pipec.h as entry point
    into the library.
---------------------------------------*/

#include "pipec.h"

int 
makedesc (char longc[], char shortc[], char units[], int type, int width, int sort)
{

   extern int Descriptors_written;
   int i,at,end;

   extern DESC Desc[];
   extern int Ndesc,Next_start;

   if (Descriptors_written) return (-1);  /* Too late now!! */
   at=Ndesc;
   /* Store information, padding or truncating as neccessary */

   if(at+1> MAXCOLS) return(-2);   /* No can do */
   switch(type)	/* determine field width */
   {
     case 'A':           /* Alphanumeric type */
           if((width % sizeof(int)) != 0) {	/* Make a mult. of int */
              width += sizeof(int) - (width % sizeof(int));
           }
	   break;
     case 'D':
     case 'T':
	   width = sizeof(double);
	   break;
     case 'R':
	   width = sizeof(float);
	   break;
     case 'I':
	   width = sizeof(int);
	   break;
      default:		/* Invalid */
	return(-3);
   }
   if((Next_start+width) > MAXLENGTH) return(-4);  /* No can do */

/* Long column name */

   end=strlen(longc);
   if(end>32) end=32;
   for(i=0;i<end;i++) Desc[at].longcol[i]=longc[i];
   for(i=end;i<32;i++) Desc[at].longcol[i]=' ';  /* pad with spaces */
   Desc[at].longcol[32]='\0';

/* Short column name field */

   end=strlen(shortc);
   if(end>8) end=8;
   for(i=0;i<end;i++) Desc[at].shortcol[i]=shortc[i];
   for(i=end;i<8;i++) Desc[at].shortcol[i]=' ';	/* pad with spaces */
   Desc[at].shortcol[8]='\0';

/* Units field */

   end=strlen(units);
   if(end>16) end=16;
   for(i=0;i<end;i++) Desc[at].units[i]=units[i];
   for(i=end;i<16;i++) Desc[at].units[i]=' ';	/* pad with spaces */
   Desc[at].units[16]='\0';

/* and the rest */

   Desc[at].type=type;
   Desc[at].width = width;
   Desc[at].sort=sort;

/* Set database parameters */

   Desc[at].start=Next_start;
   Desc[at].flags=0|OUTPUT;
   Ndesc++;					/* Set internal count */
   Next_start=Next_start+width;			/* Set start for next byte */
   return(Ndesc);
}

