/*---------------------------------------------------------------*/
/*   Copyright (c) 1993 Regents of the University of California  */
/*                 All Rights Reserved                           */
/*---------------------------------------------------------------*/
/*
      %W%  (UCLA, IGPP) %G%
*/
/*-- ffopen_select ------------------------------------------
  Re-opens all or selective portions of a flatfile for reading
  or writing given a flat file pointer. Returns the pointer 
  passed if successful or NULL is unable to open the desired 
  component. The counter portions of the flatfile structure 
  are not altered unless the data component of the flatfile 
  is opened.

  A component flatfile has each portion of a flatfile in 
  a seperate file. The descriptor portion is the definition of the
  structure that all flatfiles refered to in the data file must also
  have.

  Entry Requirements:
     ffptr: A pointer to a flatfile (as returned from ffopen()).
     option: either "r" for reading, "w" for writing, "a" for
             append, or "u" for update (insert or delete). 
             If the option is "a" the flatfile must already
             exist and only the data portion and abstract portion
             of the flatfile may be appended to.
     compon: A composite flag which is specified with the
	     parameters of FF_DAT and FF_DES which
	     specifies that the ".DAT" and ".LBL"
	     file is to be opened. The special paramter FF_ALL
	     is defined to be all the files. An example composite
	     flag is:

		FF_DAT|FF_DES

	      which specifies to open the ".DAT" and ".DES" files.

  Return Value:
      A flatfile pointer, the pointer is null if the open failed.

  Development History: (ffopen_select.c)
     Begun: 9/05/89 - Todd King

  Development History: (lffopen_select.c)
     Begun: 3/15/93 - Gilbert Hyatt

  Development History: (nffopen_select.c)
  Begun: 03/17/93 - Gilbert Hyatt
  Edited:09/08/93 - Gilbert Hyatt
  Edited:09/20/93 - Gilbert HYatt
    Added support for upper and lower case exsentions.
-------------------------------------------------------------*/
#include <sys/types.h>
#include <sys/stat.h>	/* For stat() */
#include <unistd.h>	/* For access() */
#include "ffio.h"
#include "miscc.h"

FLATFILE *
nffopen_select (FLATFILE *ffptr, char option[], unsigned int compon)
{
  char tmpname[MAX_FILENAME];
  char tmpopt[10];
  unsigned long cnt;
  char buffer[FF_DESCWIDTH];
  char tmpbuffer[FF_DESCWIDTH];
  char *ptr;
  FF_DESC *desc_ptr;
  struct stat stat;
  int i;

  /* added by Gilbert */
  char         *longname, *shortname, *units;
  char         type, *ttype, *twidth, *col, *tsort;
  unsigned int width;
  char         DESline[100], temp_str[9];
  int          col_count, sort;


  /* Check validity of options */
  if(ffptr == NULL) return(NULL);

  if(strlen(option) > 1) { return(NULL); }

  switch(option[0]) {
     case 'w':
	ffptr->open_style = FF_OPEN_WRITE;
	break;
     case 'r':
	ffptr->open_style = FF_OPEN_READ;
	break;
     case 'a':
	ffptr->open_style = FF_OPEN_APPEND;
	break;
     case 'u':
	ffptr->open_style = FF_OPEN_UPDATE;
	break;
     default:
	return(NULL);
  }  /* end of switch */


  /* Only subport reading, appending, and updating of existing Flowfiles */
  if(ffptr->open_style == FF_OPEN_WRITE) return(NULL);

  /* Preset */
  /********  OPEN DESCRIPTOR FILE ******/
  if ((compon & FF_DES) && !(ffptr->open_status & FF_DES) ) {
    if (ffptr->open_style == FF_OPEN_APPEND || ffptr->open_style == FF_OPEN_UPDATE) 
         strcpy(tmpopt, "r");
    else strcpy(tmpopt, option);

    /* See if file exists */
    strcpy(tmpname, ffptr->fname);
    strcat(tmpname, ".LBL");
    if((ffptr->desc = fopen(tmpname, tmpopt)) ==NULL){
      strcpy(tmpname, ffptr->fname);
      strcat(tmpname, ".lbl");
      if((ffptr->desc = fopen(tmpname, tmpopt)) ==NULL) return(NULL);
    }

    if (! (ffptr->open_status & FF_OPENED_BEFORE)) { /*Check if loaded before.If so don't reload*/
      if (tmpopt[0] == 'r') { /* Load internal data base with values */
	
	ffptr->reclen = atoi(TokenValue(ffptr->desc, "RECORD_BYTES", 5));

	if((col = TokenValue(ffptr->desc, "COLUMNS", 10)) == NULL) return(NULL);
	col_count = atoi(col);
	for (i =0; i < col_count; i++) {
	  if((shortname = TokenValue(ffptr->desc, "NAME",  FF_SNAME)) == NULL) return(NULL);
	  if((units     = TokenValue(ffptr->desc, "UNITS", FF_UNITS)) == NULL) units = " ";
	  if((ttype     = TokenValue(ffptr->desc, "DATA_TYPE",   12)) == NULL) return(NULL);
	  if((twidth    = TokenValue(ffptr->desc, "BYTES",       4 )) == NULL) return(NULL);
	  if((tsort     = TokenValue(ffptr->desc, "SORT_ORDER",  5 )) == NULL) tsort = "0";
	  if((longname  = TokenValue(ffptr->desc, "DESCRIPTION", FF_LNAME)) == NULL) longname = "  ";
	  if(*longname  == '\0') 
	    if((longname  = GetLine(ffptr->desc, FF_LNAME))            == NULL) longname = " ";
	  
	  width = atoi(twidth);
	  type  = (ttype[0] == 'F')?'R':ttype[0];
	  sort  = atoi(tsort);

	  ffmakedesc(ffptr, longname, shortname, units, type, width, sort);
	  free(shortname);  free(longname);  free(ttype);  
	  free(twidth);     free(units);     free(twidth);
	}  /* end of for */
      }  /* end of if */

      ffptr->open_status |= FF_OPENED_BEFORE;
      rewind(ffptr->desc);
    }  /* end of if */

    ffptr->open_status |= FF_DES;
  }  /* end of if */


  /******* OPEN DATA FILE ********/
  if ((compon & FF_DAT) && !(ffptr->open_status & FF_DAT) ) {
    if(ffptr->open_style == FF_OPEN_UPDATE) strcpy(tmpopt, "r+");
    else strcpy(tmpopt, option);
    strcat(tmpopt, "b");
     
    strcpy(tmpname, ffptr->fname);
    strcat(tmpname, ".DAT");
    if((ffptr->data = fopen(tmpname, tmpopt)) == NULL) {
      strcpy(tmpname, ffptr->fname);
      strcat(tmpname, ".dat");
      if((ffptr->data = fopen(tmpname, tmpopt)) ==NULL) return(NULL);
    }
    
     if(ffptr->reclen > 0) {	/* Determine number of records */
      fstat(fileno(ffptr->data), &stat);
      ffptr->nrecs = stat.st_size / ffptr->reclen;
    }
    ffptr->cur_recnum = 0;
    if(ffptr->open_style == FF_OPEN_APPEND) ffmove_to_rec(ffptr, ffptr->nrecs);

  }

  ffptr->type        =  FF_TYPE_FLOWFILE;
  ffptr->open_status |= FF_DAT;
  return(ffptr);
}


