/*---------------------------------------------------------------*/
/*   Copyright (c) 1989 Regents of the University of California  */
/*                 All Rights Reserved                           */
/*---------------------------------------------------------------*/
/*
      %W%  (UCLA, IGPP) %G%
*/
/*-- ffopen_select ------------------------------------------
  Re-opens all or selective portions of a flatfile for reading
  or writing given a flat file pointer. Returns the pointer 
  passed if successful or NULL is unable to open the desired 
  component. The counter portions of the flatfile structure 
  are not altered unless the data component of the flatfile 
  is opened.

  A component flatfile has each portion of a flatfile in 
  a seperate file. A virtual flatfile has all the same
  components as a component flatfile. It differs in that the
  header file contains the identifier defined as FF_VIRTUAL_MAGIC_COOKIE
  and that the data file contains names of valid flatfiles
  one per line. The descriptor portion is the definition of the
  structure that all flatfiles refered to in the data file must also
  have.

  Entry Requirements:
     ffptr: A pointer to a flatfile (as returned from ffopen()).
     option: either "r" for reading, "w" for writing, "a" for
             append, or "u" for update (insert or delete). 
             If the option is "a" the flatfile must already
             exist and only the data portion and abstract portion
             of the flatfile may be appended to.
     compon: A composite flag which is specified with the
	     parameters of FF_DAT, FF_DES, FF_ABS, FF_HED which
	     specifies that the ".DAT", ".DES", ".ABS", or ".HED"
	     file is to be opened. The special paramter FF_ALL
	     is defined to be all the files. An example composite
	     flag is:

		FF_DAT|FF_DES

	      which specifies to open the ".DAT" and ".DES" files.

  Return Value:
      A flatfile pointer, the pointer is null if the open failed.

  Development History:
     Begun: 09/05/89 - Todd King
     Edit:  08/30/93 - Gilbert Hyatt 
            the flatfile extension can now be upper or lower case.
-------------------------------------------------------------*/
#include <sys/types.h>
#include <sys/stat.h>	/* For stat() */
#include <unistd.h>	/* For access() */
#include "ffio.h"

FLATFILE *
ffopen_select (FLATFILE *ffptr, char option[], unsigned int compon)
{
  char          tmpname[MAX_FILENAME], tmpopt[10];
  unsigned long cnt;
  char          buffer[FF_DESCWIDTH], tmpbuffer[FF_DESCWIDTH];
  char          *ptr;
  struct stat   stat;
  int           i, fcnt;
  char          *fname;

#define FOPEN2(NAME, POSFIX1, POSFIX2, WHERE, TMPOPT) \
  strcpy(tmpname, NAME);                              \
  strcat(tmpname, POSFIX1);                           \
  if((WHERE = fopen(tmpname, TMPOPT)) ==NULL) {       \
    strcpy(tmpname, NAME);                            \
    strcat(tmpname, POSFIX2);                         \
    if((WHERE = fopen(tmpname, TMPOPT)) ==NULL) return(NULL); \
  }


  /* Check validity of option */
  if(ffptr == NULL)      return(NULL);
  if(strlen(option) > 1) return(NULL); 

  switch(option[0]) {
     case 'w': ffptr->open_style = FF_OPEN_WRITE;  break;
     case 'r': ffptr->open_style = FF_OPEN_READ;   break;
     case 'a': ffptr->open_style = FF_OPEN_APPEND; break;
     case 'u': ffptr->open_style = FF_OPEN_UPDATE; break;
     default:  return(NULL);
  }

  /* Preset */
  if( (compon & FF_DES) && (!(ffptr->open_status & FF_DES)) ) {

    /***** READ DESCRIPTORS INFORMATION *****/
    if(ffptr->open_style ==FF_OPEN_APPEND || ffptr->open_style ==FF_OPEN_UPDATE){ 
           strcpy(tmpopt, "r");
    } else strcpy(tmpopt, option); 

    FOPEN2(ffptr->fname, ".DES", ".des", ffptr->desc, tmpopt);
    
    if(! (ffptr->open_status & FF_OPENED_BEFORE)) { /* Check if loaded
                                            before. If so don't reload */
      if(tmpopt[0] == 'r') { /* Load internal data base with values */
	cnt = 0;
	while(fgets(buffer, sizeof(buffer), ffptr->desc) != NULL) { cnt++; }
	if(cnt == 0) {
	  fclose(ffptr->desc);
	  return(NULL);
	}
	
	rewind(ffptr->desc);
	for(i = 0; i < cnt; i++) {
	  FF_DESC desc;
	  
	  fgets(buffer, sizeof(buffer), ffptr->desc);

	  ptr = buffer;	  
	  desc.longname[0] = '\0';
	  strncpy(desc.longname, ptr, FF_LNAME);
          desc.longname[FF_LNAME] = '\0';
	  ptr += FF_LNAME;
	  strncpy(desc.shortname, ptr, FF_SNAME);
          desc.shortname[FF_SNAME] = '\0';
	  ptr += FF_SNAME;
	  strncpy(desc.units, ptr, FF_UNITS);
          desc.units[FF_UNITS] = '\0';
	  ptr += FF_UNITS;
	  desc.type = *ptr;
	  ptr++;
	  strncpy(tmpbuffer, ptr, FF_WIDTH);
	  tmpbuffer[FF_WIDTH] = '\0';
	  desc.width = atoi(tmpbuffer);
	  ptr += FF_WIDTH;
	  strncpy(tmpbuffer, ptr, FF_SORT);
	  tmpbuffer[FF_SORT] = '\0';
	  desc.sort = atoi(ptr);
	  
	  ffmakedesc(ffptr, desc.longname, desc.shortname, desc.units, 
		     desc.type, desc.width, desc.sort);
	}  
      }
      ffptr->open_status |= FF_OPENED_BEFORE;
      rewind(ffptr->desc);
    }
    ffptr->open_status |= FF_DES;
  }

  /***** READ ABSTRACT INFORMATION *****/
  if((compon & FF_ABS) && (!(ffptr->open_status & FF_ABS)) ) {
    if(ffptr->open_style == FF_OPEN_UPDATE) strcpy(tmpopt, "r");
    else                                    strcpy(tmpopt, option);
    FOPEN2(ffptr->fname, ".ABS", ".abs", ffptr->abs, tmpopt);
    ffptr->open_status |= FF_ABS;
  }

  /***** READ HEADER INFORMATION *****/
  if((compon & FF_HED) && (!(ffptr->open_status & FF_HED)) ) {
    if(ffptr->open_style ==FF_OPEN_APPEND || ffptr->open_style ==FF_OPEN_UPDATE){
           strcpy(tmpopt, "r"); 
    } else strcpy(tmpopt, option);

    FOPEN2(ffptr->fname, ".HED", ".hed", ffptr->hed, tmpopt);
    fgets(buffer, sizeof(buffer), ffptr->hed);
    if(strcmp(FF_VIRTUAL_MAGIC_COOKIE, buffer) ==0) ffptr->style =FF_STYLE_VIRTUAL;
    rewind(ffptr->hed);

    ffptr->open_status |= FF_HED;
  }

  /***** READ DATA INFORMATION *****/
  if( (compon & FF_DAT) && !(ffptr->open_status & FF_DAT) ) {
    if(ffptr->open_style == FF_OPEN_UPDATE) strcpy(tmpopt, "r+");
    else                                    strcpy(tmpopt, option);
    strcat(tmpopt, "b");
    
    FOPEN2(ffptr->fname, ".DAT", ".dat", ffptr->data, tmpopt);
 
    if(ffptr->style == FF_STYLE_VIRTUAL) {
       ffptr->vinfo.fptr = ffptr->data;
       ffptr->data = NULL;
       if(ffopen_segment(ffptr) == FF_FAILURE) { return(FF_FAILURE); };
    }

    if(ffptr->reclen > 0) {	/* Determine number of records */
      fstat(fileno(ffptr->data), &stat);
      ffptr->nrecs = stat.st_size / ffptr->reclen;
    }
    ffptr->cur_recnum = 0;

    if(ffptr->open_style == FF_OPEN_APPEND) ffmove_to_rec(ffptr, ffptr->nrecs);
  }

  ffptr->type        =  FF_TYPE_HIGHER;
  ffptr->open_status |= FF_DAT;

  return(ffptr);
}

