#include "ffio.h"
#include "sql.h"
#include <string.h>

/*------------------------------------------------------
   Read a value from a flat file and load it into an SQL
   scalar value.

   Entry requirements:
       ffptr: A valid pointer to a pre-opened flatfile.
       idx  : The index of a valid column in the flatfile.
       scalar : A pointer to the scalar to load the value
		in.

   Development History:
      Begun: 10/22/93 - Todd King

   Version:
      %W%   (UCLA/IGPP)   %G%
--------------------------------------------------------*/
int 
ffSQLReadScalar (FLATFILE *ffptr, char *column, SQL_SCALAR *scalar)
{
   char *ptr;
   unsigned int	idx;

   if(ffptr == NULL) return(FALSE);
   if(column == NULL) return(FALSE);
   if(scalar == NULL) return(FALSE);

   if(!ffscol_index(ffptr, column, &idx)) return(FALSE); /* Unknown column */

   switch(ffcol_type(ffptr, idx)) {
      case FF_INT:
	 scalar->type = SQL_TYPE_INT;
	 scalar->ival = (int) ffget_col_val(ffptr, idx);
	 break;
      case FF_FLOAT:
      case FF_DOUBLE:
	 scalar->type = SQL_TYPE_DOUBLE;
	 scalar->dval = ffget_col_val(ffptr, idx);
	 break;
      case FF_TIME:
	 scalar->type = SQL_TYPE_DATE_TIME;
	 scalar->dval = ffget_col_val(ffptr, idx);
	 break;
      case FF_ALPHA:
	 scalar->type = SQL_TYPE_TEXT;
	 ptr = ffget_col_str(ffptr, idx);
	 trimspc(ptr);
	 if(scalar->text != NULL) free(scalar->text);
	 scalar->text = strdup(ptr);
	 break;
   }

   return(TRUE);
}

/*------------------------------------------------------
   Converts a field name containing wild cards into a 
   list of field names. Returns NULL if no conversion is
   possible.

   Entry requirements:
       field : A pointer to an SQL_FIELD structure containing a
	       specification of the a field name.
       df_list : The list of data files associated with each table name.

   Development History:
      Begun: 01/06/94 - Todd King

   Version:
      %W%   (UCLA/IGPP)   %G%
--------------------------------------------------------*/
SQL_FIELD *
ffSQLFieldName (SQL_FIELD *field, LIST_NODE *df_list)
{
   char		*ptr;
   SQL_FIELD	*new;
   SQL_FIELD	*list = NULL;
   FLATFILE	*ffptr;
   int 		i;
   int		n;

   if(field == NULL) return(NULL);
   if(df_list == NULL) return(NULL);

   /* Currently only the pattern "*" is supported which is
      translated into a list of fields in the flatfile.
      This might be changed in the future to support more
      complex wild carding.
   */

   if(strcmp(field->column, "*") != 0) return(NULL);

   ffptr = (FLATFILE *) list_get_data(list_find_node(df_list, field->table));
   if(ffptr == NULL) return(NULL);

   /* Scan for all matching field names in flatfile */

   n = ffncol(ffptr);
   for(i = 0; i < n; i++) {
      new = sql_make_field();
      if(new != NULL) {
	 new->column = ffcol_sname(ffptr, i);
	 new->table = strdup(field->table);
	 list = sql_push_field(list, new);
      }
   }

   return(list);
}
