/*
      @(#)pvrot.c	1.1         (UCLA/IGPP)      9/26/95
*/
/*--------------------------------------------------------------------

------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h> 
#include <math.h>
#include "highpipec.h"
#include "cmdlinec.h"
#include "ffio.h"
#include "time.h"

#ifdef W32
#include "binary.h"
#endif


#ifndef W32
#include "BC.h"
#endif

#define  TRUE 1
#define  FALSE 0
#define  SUCCESS  1
#define  FAILURE  0

char *Cl_options[] = {"CD_LOCATION", "ORBIT", "OUTFILE", "HOST",  "FILETYPE",  ""};
enum {CD_LOCATION = 1, ORBIT, OUTFILE, HOST, FILETYPE}; 

char *Host_types[] = {"SUN", "HP", "VAX",  "PC", ""};
enum {SUN = 0, HP, VAX,  PC};

char *File_types[] = {"ASCII", "FLATFILE", "FLOW", ""};
enum {ASCII = 0, FLAT, FLOW};

 
typedef struct {
     FILE     *magfptr;
     FILE     *ephfptr;
     FILE     *ofptr;
     FILE     *hfptr;
     FILE     *lblptr;
     int       file_type;
     int       num_recs;
     int       first_orb;
     int       last_orb;
     int       host;
     int       orbit;
     char      cd_location[512];
     char      magfile_name[512];
     char      ephfile_name[512];
     char      outfile_name[512];
     char      hdrfile_name[512];
     char      lblfile_name[512];
     char      product_id[512];
     char      mag_path[512];
     char      ephem_path[512];
     char      start_time[24];
     char      stop_time[24];
} Options;
 
typedef struct {
     double   time;
     float    x_vso;
     float    y_vso;
     float    z_vso;
     float    alt;
     float    sza;
     float    plong;
     float    plat;
     float    spin_x;
     float    spin_y;
     float    spin_z;
     float    clat;
     float    clong;
     float    elong;
     float    rsun;
} Ephem;
#define SIZE_OF_EPHEM   64
 
typedef struct {
     double   time;
     float    Bx;
     float    By;
     float    Bz;
     float    Bt;
} Mag;
#define SIZE_OF_MAG     24

#define ERR_INVALID_HOST        0
#define ERR_NO_ORBIT	        1
#define ERR_NO_OPTION           2
#define ERR_OUTPUT_OPEN_FAIL    3
#define ERR_INPUT_OPEN_FAIL     4
#define ERR_FILE_READ           5
#define ERR_BAD_PATH            6
#define ERR_NO_FLOW             7
#define ERR_INVALID_FILETYPE    8
#define NUM_ERRORS              9


/*------------------------------------------------
   Error message and exit handler function.
--------------------------------------------------*/
void msg_exit(
   int   err_code,
   char *filler)
{
   char *error[NUM_ERRORS];
   char *fitting;

   fitting = "pvrot";

   error[ERR_INVALID_HOST] = "Bad value for HOST option '@' - fatal!.";
   error[ERR_NO_ORBIT]  = "'@', fatal!";
   error[ERR_NO_OPTION] = "'@' - fatal!";
   error[ERR_OUTPUT_OPEN_FAIL] = "Unable to open file '@' - fatal!";
   error[ERR_INPUT_OPEN_FAIL] = "Unable to open file '@' - fatal!";
   error[ERR_FILE_READ] = "'@' - fatal!";
   error[ERR_BAD_PATH] = "CDROM location '@' invalid - fatal!";
   error[ERR_INVALID_FILETYPE] = "FILETYPE '@' is unknown, default is ASCII.";
   fillerrexit(err_code, NUM_ERRORS, fitting, error, filler);
}
/*------------------------------------------------
   Error message and exit handler function.
--------------------------------------------------*/
void msg_exit2(
   int   err_code,
   char *filler)
{

   switch(err_code) {
      case ERR_INVALID_HOST: 
         fprintf(stderr, "Bad value for HOST option %s - fatal!\n", filler);
         exit(3);
         break;
      case ERR_NO_ORBIT:
         fprintf(stderr, "%s, fatal!\n", filler);
         exit(3);
         break;
      case ERR_NO_OPTION:
         fprintf(stderr, "%s - fatal!\n", filler);
         exit(3);
         break;
      case ERR_OUTPUT_OPEN_FAIL:
         fprintf(stderr, "Unable to open file %s - fatal!\n", filler);
         fprintf(stderr, "Probable failure reason: no write permission\n");
         exit(3);
         break;
      case ERR_INPUT_OPEN_FAIL:
         fprintf(stderr, "Unable to open file %s - fatal!\n", filler);
         fprintf(stderr, "Probable failure reason: orbit not on CDROM \n");
         exit(3);
         break;
      case ERR_FILE_READ:
         fprintf(stderr, "%s - fatal!\n", filler);
     fprintf(stderr, "Probable failure reason: CDROM location not specified\n");
         fprintf(stderr, "and code not run from proper CDROM directory. \n");
         exit(3);
         break;
      case ERR_BAD_PATH:
         fprintf(stderr, "CDROM location %s invalid - fatal!\n", filler);
         exit(3);
         break;
      case ERR_NO_FLOW:
         fprintf(stderr, "%s - default is ASCII\n", filler);
         exit(3);
         break;
      case ERR_INVALID_FILETYPE:
         fprintf(stderr, "FILETYPE %s is unknown, default is ASCII.\n", filler);
         break;
   }
}

/*-----------------------------------------------------------------------
  PickVarFormat() - selects the right conversion functions for the
     desired format.

  Written by  - Gilbert Hyatt
------------------------------------------------------------------------*/
int
PickVarFormat (int host_type)
{
  switch(host_type) {
    /*  Convert to VAX/VMS Format  */
    case VAX: BCVAXFormat();
              break;
    /*  Convert to HP 1000 Format  */
    case HP: BCHP1000Format();
              break;
    /*  Convert to Intel PC using IEEE Format  */
    case PC: BCPCFormat();
              break;
    /*  Convert to Sun Format      */
    case SUN: BCSunFormat();
              break;
    /*  Non of the above           */
    default:
      return(0);
      break;
  }
  return(1);
}
/*-- maxdaymon ---------------------------------------------------
 Returns the number of days in a month given the index of the
 month and the year.

 Entry Requirements:
    mon: The index of the month (1 = January, 12 = December)
    yr: The full year specification. e.g 1987.

  Return Value:
    The number of days in the month.

   Version:
       @(#)daymon.c     1.2   (UCLA/IGPP)   16 Jun 1995
----------------------------------------------------------------*/
int maxdaymon(
   int mon,                     /* Jan = 1, Feb = 2, etc */
   int yr                               /* Full year A.D., e.g., 1987 */
) {
  int i;

  i = daymon[mon-1];
  if((mon == 2) && !(yr & 0x3) && (yr != 2000)) i++; /* chk leap yr & february */
  return(i);
}

/*-- secyr ----------------------------------------------------
  Calculates the numbero f seconds in a given year.

  Entry Requirements:
    yr: The year.

  Return Value:
    The numbero f seconds in the year.

   Version:
       @(#)secperyr.c   1.2   (UCLA/IGPP)   16 Jun 1995
-------------------------------------------------------------*/
double
secperyr(
   int yr
) {
  double x;
  x = 86400.0*365.0;
  if(!(yr & 0x3) && (yr != 2000)) x += 86400.0;
  return(x);
}

/*--yrmonday ----------------------------------------------
 Calculates the year, month, day, hour, minutes and seconds
 given the floating point seconds from Jan 1, in the
 reference year (1966)

  Entry Requirements:
    sectime: The number of seconds from the beginning of the
           reference year.
    yr: Assigned the year.
    mon: Assigned the month.
    day: Assigned the day of the month.
    hr: Assigned the hour or the day.
    min: Assigned the minutes of the hour.
    sec: Assigned the seconds and milliseconds of the minute.

  Return Value:
    undefined.

   Version:
       @(#)yrmonday.c   1.4    (UCLA/IGPP)   16 Jun 1995
---------------------------------------------------------*/
int yrmonday(
   double sectime,
   int *yr,
   int *mon,
   int *day,
   int *hr,
   int *min,
   float *sec
) {
  double secpermo, secsyr;
  double fmod(), secperyr();

  *yr = 1966;
               /* Year 2100 */           /* Year 2101 - invalid time */
  if(sectime > 4228675200.000) sectime = 4260297600.000;
  while(sectime > (secsyr = secperyr(*yr)) - 0.5e-3) {
    sectime -= secsyr;
    *yr += 1;
  }
  *mon = 1;
  while(sectime > (secpermo = 86400.0*maxdaymon(*mon,*yr)) - 0.5e-3) {
    sectime -= secpermo;
    *mon += 1;
    if(*mon == 13) {*mon = 1; *yr += 1;}
  }
  if(sectime < 0) sectime = 0;
  *day = (int) (sectime/86400 + 1);
  sectime = fmod(sectime,86400.0);
  *hr = (int) (sectime/3600.0);
  sectime = fmod(sectime,3600.0);
  *min = (int) (sectime/60.0);
  sectime = fmod(sectime,60.0);
  *sec = (float) sectime;
  return(1);
}

/*--------------------------------------------------------------
   Returns the array index of the search character within the search string.

   Entry Requirements:
      search_str : A pointer to the search string.
      search_char : Character to search for.

   Return Values:
      Last location of the search character within the search string or
      -1 if the search character is not found.
----------------------------------------------------------------*/
int
loc_char(
  char *search_str,
  char  search_char
) {

  int i;

  for (i=strlen(search_str); i>= 0; i--) {
     if (search_str[i] == search_char) return(i);
  }
  return(-1);
}

/*--------------------------------------------------------------
   Calls the requested data writing function
 
   Entry Requirements:
      fname : A point tox the data writing function.
 
   Return Values:
      none
----------------------------------------------------------------*/
void write_data(
   Options *options,
   Mag     *mag,
   int    (*fname)())
{
   fname(options, mag);
}
 
/*--------------------------------------------------------------
   Calls the requested PDS label writing function
 
   Entry Requirements:
      fname : A point to the PDS label writing function.
 
   Return Values:
      none
----------------------------------------------------------------*/
int write_label(
   Options *options,
   int    (*fname)())
{
   fname(options);
   return(SUCCESS);
}

/*--------------------------------------------------------------
   Converts a clinetime into a PDS time string. 

   Entry Requirements:
      time:      Clinetime - double precission secs sine 1966.

   Return Values:
      Ptr to a time string.
----------------------------------------------------------------*/
int  pdstime(
  double     time,
  char       *timestr)
{
  int yr, mon, day;
  int hour, min;
  float sec;

  yrmonday(time, &yr, &mon, &day, &hour, &min, &sec);
  sprintf(timestr,"%4d-%02d-%02dT%02d:%02d:%06.3f\0", 
    yr, mon, day, hour, min, sec);
  return(SUCCESS);
}

/*--------------------------------------------------------------
   Writes ASCII mag data to the output file
 
   Entry Requirements:
      options:  A pointer to the structure that contains the output file ptr.
      mag:      A pointer to the structure that contains the data. 
   Return Values:
      none
----------------------------------------------------------------*/
int ascii_data_writer(
   Options *options,
   Mag     *mag)
{
  char   timestr[24];

  pdstime(mag->time, timestr);
  if (mag->Bx > 1.0e10) mag->Bx =  9999.9999;
  if (mag->By > 1.0e10) mag->By =  9999.9999;
  if (mag->Bz > 1.0e10) mag->Bz =  9999.9999;
  if (mag->Bt > 1.0e10) mag->Bt =  9999.9999;
  fprintf(options->ofptr ,"%24s %10.4f %10.4f %10.4f %10.4f\n",
   timestr, mag->Bx,  mag->By, mag->Bz, mag->Bt);
  return(SUCCESS);
}



/*--------------------------------------------------------------
   Writes mag data to the output flatfile
 
   Entry Requirements:
      options:  A pointer to the structure that contains the flatfile ptr.
      mag:      A pointer to the structure that contains the data. 
   Return Values:
      none
----------------------------------------------------------------*/
int ff_data_writer(
   Options *options,
   Mag     *mag)
{
  if (mag->Bx > 1.0e10) mag->Bx =  1.0e32;
  if (mag->By > 1.0e10) mag->By =  1.0e32;
  if (mag->Bz > 1.0e10) mag->Bz =  1.0e32;
  if (mag->Bt > 1.0e10) mag->Bt =  1.0e32;
  fwrite((char *)mag, sizeof(Mag), 1, options->ofptr);
  return(SUCCESS);
} 

/*--------------------------------------------------------------
   Writes mag data to the data flow
 
   Entry Requirements:
      options:  A pointer to the options structure.
      mag:      A pointer to the structure that contains the data. 
   Return Values:
      none
----------------------------------------------------------------*/
int flow_data_writer(
   Options *options,
   Mag     *mag)
{
#ifndef WIN32
    if (mag->Bx > 1.0e10 || mag->By > 1.0e10 || mag->Bz > 1.0e10) {
       setcolval("time",mag->time);
       setcolflag("Bx",mag->Bx);
       setcolflag("By",mag->By);
       setcolflag("Bz",mag->Bz);
       setcolflag("Bt",mag->Bt);
    } else {
       setcolval("time",mag->time);
       setcolval("Bx",mag->Bx);
       setcolval("By",mag->By);
       setcolval("Bz",mag->Bz);
       setcolval("Bt",mag->Bt);
    }
    wdata();
#endif
	return(SUCCESS);
} 

/*--------------------------------------------------------------
   Writes the ffh file for a flatfile.

   Entry Requirements:
      options: Pointer to the options structure.
 
   Return Values:
      none
----------------------------------------------------------------*/
int write_ff_hdr(
   Options *options)
{
   char outfile[1024];
   char infile[1024];
   char aline[73];
   int  slen;
   int  i,j;
   int  cnt;
   FILE *ifptr;


   strcpy(infile, options->magfile_name);
   slen = strlen(infile);

   if (infile[slen-1] == 'd') infile[slen-1] = 'h';
   if (infile[slen-1] == 'D') infile[slen-1] = 'H';
   
   ifptr = fopen(infile, "r");
   cnt = 1;
   strcpy(outfile, options->product_id);
   fgets(aline, 73, ifptr);
   strcpy(aline, "DATA  = ");
   strcat(aline, outfile);
   
   fprintf(options->hfptr,"%s",aline);
   slen = strlen(aline);
   for (i=slen; i<72; i++) fprintf(options->hfptr,"%c", ' ');
   for (i=1;i<8; i++) {
      fgets(aline, 73, ifptr);
      slen = strlen(aline);
      fprintf(options->hfptr, "%s", aline);
      for (j=slen; j<72; j++) fprintf(options->hfptr,"%c", ' ');
      strcpy(aline, "");
   }
   for (i=0;i<3; i++) fgets(aline, 73, ifptr);
   fprintf(options->hfptr, "002 BX VSO    NT        PVO MAG                   R       8             "); 
   fprintf(options->hfptr, "003 BY VSO    NT        PVO MAG                   R      12             ");
   fprintf(options->hfptr, "004 BZ VSO    NT        PVO MAG                   R      16             ");
   for (i=0;i<12; i++) {
      fgets(aline, 73, ifptr);
      slen = strlen(aline);
      fprintf(options->hfptr, "%s", aline);
      for (j=slen; j<72; j++) fprintf(options->hfptr,"%c", ' ');
      strcpy(aline, "");
   }

fprintf(options->hfptr, "PVROT:                                                                  ");
fprintf(options->hfptr, "  PVO magnetic field rotated from spacecraft to VSO coordinates.        ");
fprintf(options->hfptr, "END                                                                     ");
fclose (options->hfptr);
fclose (ifptr);
return(SUCCESS);
 
}


/*--------------------------------------------------------------
   Writes the PDS label for a flatfile.
 
   Entry Requirements:
      options: Pointer to the options structure.
 
   Return Values:
      none
----------------------------------------------------------------*/
int lbl_ff_data(
   Options *options)
{
   char outfile[1024];

strcpy(outfile, options->product_id);

fprintf(options->lblptr,"CCSD3ZF0000100000001NJPL3IF0PDSX00000001\n");
fprintf(options->lblptr,"PDS_VERSION_ID                = PDS3\n");
fprintf(options->lblptr,"RECORD_TYPE                   = FIXED_LENGTH\n");
fprintf(options->lblptr,"RECORD_BYTES                  = 24\n");
fprintf(options->lblptr,"FILE_RECORDS                  = %d\n", options->num_recs);
fprintf(options->lblptr,"^TIME_SERIES                  = \"%s\"\n",outfile);
fprintf(options->lblptr,"DATA_SET_ID                   = \"PVO-V-OMAG-3--SCCOORDS-HIRES-V1.0\"\n");
fprintf(options->lblptr,"SPACECRAFT_NAME               = \"PIONEER VENUS ORBITER\"\n");
fprintf(options->lblptr,"INSTRUMENT_NAME               = \"FLUXGATE MAGNETOMETER\"\n");
fprintf(options->lblptr,"TARGET_NAME                   = \"VENUS\"\n");
fprintf(options->lblptr,"START_TIME                    = %s\n",options->start_time);
fprintf(options->lblptr,"STOP_TIME                     = %s\n",options->stop_time);
fprintf(options->lblptr,"MISSION_PHASE_NAME            = \"VENUS ORBITAL OPERATIONS\"\n");
fprintf(options->lblptr,"PRODUCT_ID                    = \"%s\"\n",outfile);
fprintf(options->lblptr,"PRODUCT_CREATION_TIME         = 1995-10-01\n");
fprintf(options->lblptr,"SPACECRAFT_CLOCK_START_COUNT  = \"UNK\"\n");
fprintf(options->lblptr,"SPACECRAFT_CLOCK_STOP_COUNT   = \"UNK\"\n");
fprintf(options->lblptr,"OBJECT                        = TIME_SERIES\n");
fprintf(options->lblptr,"  INTERCHANGE_FORMAT            = BINARY\n");
fprintf(options->lblptr,"  ROWS                          = %d\n", options->num_recs);
fprintf(options->lblptr,"  COLUMNS                       = 5\n");
fprintf(options->lblptr,"  ROW_BYTES                     = 24\n");
fprintf(options->lblptr,"  SAMPLING_PARAMETER_NAME       = TIME\n");
fprintf(options->lblptr,"  SAMPLING_PARAMETER_UNIT       = SECOND\n");
fprintf(options->lblptr,"  SAMPLING_PARAMETER_INTERVAL   = \"N/A\"\n");
fprintf(options->lblptr,"  OBJECT                        = COLUMN\n");
fprintf(options->lblptr,"    NAME                          = \"TIME\"\n");
fprintf(options->lblptr,"    COLUMN_NUMBER                 = 1\n");
fprintf(options->lblptr,"    UNIT                          = \"SEC\"\n");
fprintf(options->lblptr,"    DATA_TYPE                     = \"FLOAT\"\n");
fprintf(options->lblptr,"    START_BYTE                    = 1\n");
fprintf(options->lblptr,"    BYTES                         = 8\n");
fprintf(options->lblptr,"    DESCRIPTION                   = \"Universal Time given in seconds since\n");
fprintf(options->lblptr,"    1966-01-01T00:00:00.000 where no attempt has been made to account for\n");
fprintf(options->lblptr,"    leap seconds.\"\n");
fprintf(options->lblptr,"  END_OBJECT                    = COLUMN\n");
fprintf(options->lblptr,"  OBJECT                        = COLUMN\n");
fprintf(options->lblptr,"    NAME                          = \"BX VSO\"\n");
fprintf(options->lblptr,"    COLUMN_NUMBER                 = 2\n");
fprintf(options->lblptr,"    UNIT                          = \"NT\"\n");
fprintf(options->lblptr,"    DATA_TYPE                     = \"FLOAT\"\n");
fprintf(options->lblptr,"    START_BYTE                    = 9\n");
fprintf(options->lblptr,"    BYTES                         = 4\n");
fprintf(options->lblptr,"    DESCRIPTION                   = \"BX VSO: The X-component of the magnetic \n");
fprintf(options->lblptr,"    field in Venus Solar Orbital (VSO) coordinates. The VSO X axis points along\n");
fprintf(options->lblptr,"    the Venus - Sun line, positive towards the Sun.\"\n");
fprintf(options->lblptr,"  END_OBJECT                    = COLUMN\n");
fprintf(options->lblptr,"  OBJECT                        = COLUMN\n");
fprintf(options->lblptr,"    NAME                          = \"BY VSO\"\n");
fprintf(options->lblptr,"    COLUMN_NUMBER                 = 3\n");
fprintf(options->lblptr,"    UNIT                          = \"NT\"\n");
fprintf(options->lblptr,"    DATA_TYPE                     = \"FLOAT\"\n");
fprintf(options->lblptr,"    START_BYTE                    = 13\n");
fprintf(options->lblptr,"    BYTES                         = 4\n");
fprintf(options->lblptr,"    DESCRIPTION                   = \"BY VSO: The Y-component of the magnetic \n");
fprintf(options->lblptr,"    field in Venus Solar Orbital (VSO) coordinates. The VSO Y axis is lies\n");
fprintf(options->lblptr,"    in Venus' orbial plane and points in the directions opposing planetary \n");
fprintf(options->lblptr,"    motion.\"\n");
fprintf(options->lblptr,"  END_OBJECT                    = COLUMN\n");
fprintf(options->lblptr,"  OBJECT                        = COLUMN\n");
fprintf(options->lblptr,"    NAME                          = \"BZ VSO\"\n");
fprintf(options->lblptr,"    COLUMN_NUMBER                 = 4\n");
fprintf(options->lblptr,"    UNIT                          = \"NT\"\n");
fprintf(options->lblptr,"    DATA_TYPE                     = \"FLOAT\"\n");
fprintf(options->lblptr,"    START_BYTE                    = 17\n");
fprintf(options->lblptr,"    BYTES                         = 4\n");
fprintf(options->lblptr,"    DESCRIPTION                   = \"BZ VSO: The Z-component of the magnetic \n");
fprintf(options->lblptr,"    field in Venus Solar Orbital (VSO) coordinates. The VSO Z axis is\n");
fprintf(options->lblptr,"    defined to be parallel to the upward normal of the Venus \n");
fprintf(options->lblptr,"    orbital plane.\"\n");
fprintf(options->lblptr,"  END_OBJECT                    = COLUMN\n");
fprintf(options->lblptr,"  OBJECT                        = COLUMN\n");
fprintf(options->lblptr,"    NAME                          = \"BT\"\n");
fprintf(options->lblptr,"    COLUMN_NUMBER                 = 5\n");
fprintf(options->lblptr,"    UNIT                          = \"NT\"\n");
fprintf(options->lblptr,"    DATA_TYPE                     = \"FLOAT\"\n");
fprintf(options->lblptr,"    START_BYTE                    = 21\n");
fprintf(options->lblptr,"    BYTES                         = 4\n");
fprintf(options->lblptr,"    DESCRIPTION                   = \"BT: The vector magnitude of the magnetic \n");
fprintf(options->lblptr,"    field.\"\n");
fprintf(options->lblptr,"  END_OBJECT                    = COLUMN\n");
fprintf(options->lblptr,"END_OBJECT                    = TIME_SERIES\n");
fprintf(options->lblptr,"END\n");
return(SUCCESS);
 }

/*--------------------------------------------------------------
   Writes the PDS labels for an ASCII data file
 
   Entry Requirements:
      options: Pointer to the options structure.
 
   Return Values:
      none
----------------------------------------------------------------*/
lbl_ascii_data(
   Options *options)
{
   char outfile[1024];

strcpy(outfile, options->product_id);

fprintf(options->lblptr,"CCSD3ZF0000100000001NJPL3IF0PDSX00000001\n");
fprintf(options->lblptr,"PDS_VERSION_ID                = PDS3\n");
fprintf(options->lblptr,"RECORD_TYPE                   = FIXED_LENGTH\n");
fprintf(options->lblptr,"RECORD_BYTES                  = 68\n");
fprintf(options->lblptr,"FILE_RECORDS                  = %d\n",options->num_recs);
fprintf(options->lblptr,"^TIME_SERIES                  = \"%s\"\n",outfile);
fprintf(options->lblptr,"DATA_SET_ID                   = \"PVO-V-OMAG-3--SCCOORDS-HIRES-V1.0\"\n");
fprintf(options->lblptr,"SPACECRAFT_NAME               = \"PIONEER VENUS ORBITER\"\n");
fprintf(options->lblptr,"INSTRUMENT_NAME               = \"FLUXGATE MAGNETOMETER\"\n");
fprintf(options->lblptr,"TARGET_NAME                   = \"VENUS\"\n");
fprintf(options->lblptr,"START_TIME                    = %s\n",options->start_time);
fprintf(options->lblptr,"STOP_TIME                     = %s\n",options->stop_time);
fprintf(options->lblptr,"MISSION_PHASE_NAME            = \"VENUS ORBITAL OPERATIONS\"\n");
fprintf(options->lblptr,"PRODUCT_ID                    = \"%s\"\n",outfile);
fprintf(options->lblptr,"PRODUCT_CREATION_TIME         = 1995-10-01\n");
fprintf(options->lblptr,"SPACECRAFT_CLOCK_START_COUNT  = \"UNK\"\n");
fprintf(options->lblptr,"SPACECRAFT_CLOCK_STOP_COUNT   = \"UNK\"\n");
fprintf(options->lblptr,"OBJECT                        = TIME_SERIES\n");
fprintf(options->lblptr,"  INTERCHANGE_FORMAT            = ASCII\n");
fprintf(options->lblptr,"  ROWS                          = %d\n",options->num_recs);
fprintf(options->lblptr,"  COLUMNS                       = 5\n");
fprintf(options->lblptr,"  ROW_BYTES                     = 68\n");
fprintf(options->lblptr,"  SAMPLING_PARAMETER_NAME       = TIME\n");
fprintf(options->lblptr,"  SAMPLING_PARAMETER_UNIT       = SECOND\n");
fprintf(options->lblptr,"  SAMPLING_PARAMETER_INTERVAL   = \"N/A\"\n");
fprintf(options->lblptr,"  OBJECT                        = COLUMN\n");
fprintf(options->lblptr,"    NAME                          = \"TIME\"\n");
fprintf(options->lblptr,"    COLUMN_NUMBER                 = 1\n");
fprintf(options->lblptr,"    UNIT                          = \"SEC\"\n");
fprintf(options->lblptr,"    DATA_TYPE                     = \"CHARACTER\"\n");
fprintf(options->lblptr,"    START_BYTE                    = 1\n");
fprintf(options->lblptr,"    BYTES                         = 24\n");
fprintf(options->lblptr,"    DESCRIPTION                   = \"Universal Time given in PDS\n");
fprintf(options->lblptr,"    time format YYYY-MM-DDThh:mm:ss.sss\"\n");
fprintf(options->lblptr,"  END_OBJECT                    = COLUMN\n");
fprintf(options->lblptr,"  OBJECT                        = COLUMN\n");
fprintf(options->lblptr,"    NAME                          = \"BX VSO\"\n");
fprintf(options->lblptr,"    COLUMN_NUMBER                 = 2\n");
fprintf(options->lblptr,"    UNIT                          = \"NT\"\n");
fprintf(options->lblptr,"    DATA_TYPE                     = \"FLOAT\"\n");
fprintf(options->lblptr,"    START_BYTE                    = 25\n");
fprintf(options->lblptr,"    BYTES                         = 11\n");
fprintf(options->lblptr,"    DESCRIPTION                   = \"BX VSO: The X-component of the magnetic \n");
fprintf(options->lblptr,"    field in Venus Solar Orbital (VSO) coordinates. The VSO X axis points along\n");
fprintf(options->lblptr,"    the Venus - Sun line, positive towards the Sun.\"\n");
fprintf(options->lblptr,"  END_OBJECT                    = COLUMN\n");
fprintf(options->lblptr,"  OBJECT                        = COLUMN\n");
fprintf(options->lblptr,"    NAME                          = \"BY VSO\"\n");
fprintf(options->lblptr,"    COLUMN_NUMBER                 = 3\n");
fprintf(options->lblptr,"    UNIT                          = \"NT\"\n");
fprintf(options->lblptr,"    DATA_TYPE                     = \"FLOAT\"\n");
fprintf(options->lblptr,"    START_BYTE                    = 36\n");
fprintf(options->lblptr,"    BYTES                         = 11\n");
fprintf(options->lblptr,"    DESCRIPTION                   = \"BY VSO: The Y-component of the magnetic \n");
fprintf(options->lblptr,"    field in Venus Solar Orbital (VSO) coordinates. The VSO Y axis is lies\n");
fprintf(options->lblptr,"    in Venus' orbial plane and points in the directions opposing planetary \n");
fprintf(options->lblptr,"    motion.\"\n");
fprintf(options->lblptr,"  END_OBJECT                    = COLUMN\n");
fprintf(options->lblptr,"  OBJECT                        = COLUMN\n");
fprintf(options->lblptr,"    NAME                          = \"BZ VSO\"\n");
fprintf(options->lblptr,"    COLUMN_NUMBER                 = 4\n");
fprintf(options->lblptr,"    UNIT                          = \"NT\"\n");
fprintf(options->lblptr,"    DATA_TYPE                     = \"FLOAT\"\n");
fprintf(options->lblptr,"    START_BYTE                    = 47\n");
fprintf(options->lblptr,"    BYTES                         = 11\n");
fprintf(options->lblptr,"    DESCRIPTION                   = \"BZ VSO: The Z-component of the magnetic \n");
fprintf(options->lblptr,"    field in Venus Solar Orbital (VSO) coordinates. The VSO Z axis is\n");
fprintf(options->lblptr,"    defined to be parallel to the upward normal of the Venus \n");
fprintf(options->lblptr,"    orbital plane.\"\n");
fprintf(options->lblptr,"  END_OBJECT                    = COLUMN\n");
fprintf(options->lblptr,"  OBJECT                        = COLUMN\n");
fprintf(options->lblptr,"    NAME                          = \"BT\"\n");
fprintf(options->lblptr,"    COLUMN_NUMBER                 = 5\n");
fprintf(options->lblptr,"    UNIT                          = \"NT\"\n");
fprintf(options->lblptr,"    DATA_TYPE                     = \"FLOAT\"\n");
fprintf(options->lblptr,"    START_BYTE                    = 58\n");
fprintf(options->lblptr,"    BYTES                         = 11\n");
fprintf(options->lblptr,"    DESCRIPTION                   = \"BT: The vector magnitude of the magnetic \n");
fprintf(options->lblptr,"    field.\"\n");
fprintf(options->lblptr,"  END_OBJECT                    = COLUMN\n");
fprintf(options->lblptr,"END_OBJECT                    = TIME_SERIES\n");
fprintf(options->lblptr,"END\n");
return(SUCCESS);
 
}
/*--------------------------------------------------------------
   Converts the required data from the ephemeris structure to the native
   floating point form.
 
   Entry Requirements:  
      options:  Pointer to the structure containing the host information.
      ephem:    Pointer to the empheris data structure.
 
   Return Values:                                       
      None
----------------------------------------------------------------*/
int
convert_ephem (
   Options *options,
   Ephem *ephem)
{
 BCIEEEToDouble(&ephem->time);
 BCIEEEToFloat( &ephem->spin_x);
 BCIEEEToFloat( &ephem->spin_y);
 BCIEEEToFloat( &ephem->spin_z);
return(SUCCESS);
}
 /*--------------------------------------------------------------
   Converts the data from the mag structure to the native
   floating point form.
 
   Entry Requirements:           
      options:  Pointer to the structure containing the host information.
      mag:      Pointer to the mag data structure.
 
   Return Values:                        
      None
----------------------------------------------------------------*/
int
convert_mag (
   Options *options,
   Mag *mag)
{
/* Set output binary form to Host Type  */
 
 BCIEEEToDouble(&mag->time);
 BCIEEEToFloat( &mag->Bx);
 BCIEEEToFloat( &mag->By);
 BCIEEEToFloat( &mag->Bz);
 BCIEEEToFloat( &mag->Bt);
 return(SUCCESS);
}
/*--------------------------------------------------------------
   Set the paths to the input data files for POSIX systems.
 
   Entry Requirements:           
      options:  Pointer to the options structure
 
   Return Values:                        
     mag_path:     Character string containing the mag data path
     epphem_path:  Character string containing the ephemeris data path
----------------------------------------------------------------*/
int
posix_data_path (
   Options *options)
{
   int i;

   i = strlen(options->cd_location);
   if( i <= 0) {
/*
      strcpy(options->mag_path, "../../../DATA/OMAG/HIRES/");
      strcpy(options->ephem_path, "../../../DATA/EPHEM/");
*/
      strcpy(options->mag_path, "../../../data/omag/hires/");
      strcpy(options->ephem_path, "../../../data/ephem/");
   } else {
      if (options->cd_location[strlen(options->cd_location)] != '/')
         strcat(options->cd_location, "/");
      strcpy(options->mag_path, options->cd_location);
      strcpy(options->ephem_path, options->cd_location);
/*
      strcat(options->mag_path, "DATA/OMAG/HIRES/");
      strcat(options->ephem_path, "DATA/EPHEM/");
*/
      strcat(options->mag_path, "data/omag/hires/");
      strcat(options->ephem_path, "data/ephem/");
	  return(SUCCESS);
   }
}
/*--------------------------------------------------------------
   Set the paths to the input data files for VMS systems.
 
   Entry Requirements:           
      options:  Pointer to the options structure
 
   Return Values:                        
     mag_path:     Character string containing the mag data path
     epphem_path:  Character string containing the ephemeris data path
----------------------------------------------------------------*/
int
vms_data_path (
   Options *options)
{
   int i;

   i = strlen(options->cd_location);
   if( i <= 0) {
      strcpy(options->mag_path, "[-.-.-.DATA.OMAG.HIRES]");
      strcpy(options->ephem_path, "[-.-.-.DATA.EPHEM]");
   } else {
      if (options->cd_location[strlen(options->cd_location) - 1] == ':') {
         strcpy(options->mag_path, options->cd_location);
         strcpy(options->ephem_path, options->cd_location);
         strcat(options->mag_path, "[DATA.OMAG.HIRES]");
         strcat(options->ephem_path, "[DATA.EPHEM]");
      }
      if (options->cd_location[strlen(options->cd_location) - 1] == ']') {
         strncpy(options->mag_path, options->cd_location, i - 1);
         strcpy(options->ephem_path, options->mag_path);
         strcat(options->mag_path, ".DATA.OMAG.HIRES]");
         strcat(options->ephem_path, ".DATA.EPHEM]");
      }
   }
   return(SUCCESS);
 }
/*--------------------------------------------------------------
   Set the paths to the input data files for MS-DOS systems.
 
   Entry Requirements:           
      options:  Pointer to the options structure
 
   Return Values:                        
     mag_path:     Character string containing the mag data path
     epphem_path:  Character string containing the ephemeris data path
----------------------------------------------------------------*/
int
msdos_data_path (
   Options *options)
{
   int i;

   i = strlen(options->cd_location);
   if( i <= 0) {
      strcpy(options->mag_path, "..\\..\\..\\DATA\\OMAG\\HIRES\\");
      strcpy(options->ephem_path, "..\\..\\..\\DATA\\EPHEM\\");
   } else {
      if (options->cd_location[strlen(options->cd_location)-1] != '\\')
		  strcat(options->cd_location, "\\");
      strcpy(options->mag_path, options->cd_location);
      strcpy(options->ephem_path, options->cd_location);
      strcat(options->mag_path, "DATA\\OMAG\\HIRES\\");
      strcat(options->ephem_path, "DATA\\EPHEM\\");
   }
   return(SUCCESS);
}

/*----------------------------------------------------------------
   Determines the CD start/stop orbit numbers on DOS systems.

   Entry Requirements:
      options:  Pointer to the options structure

   Return Values:
      SUCCESS/FAILURE  
----------------------------------------------------------------*/
int
   get_dos_orbs(
   Options *options)
{
   int  i;
   int  iorb;
   int  first;
   char buffer[1024];
   char tmp[1024];
   char command[1024];
   FILE *ptr;
 
   first = TRUE;
   strcpy(command, "dir ");
   strcat(command, options->mag_path);
   strcat(command, "ORB*.LBL > c:\\tmp12312.asc");
   i =  system(command);
   if (i < 0) msg_exit(ERR_BAD_PATH, options->cd_location);
   strcpy(tmp,"c:\\tmp12312.asc");
   if ((ptr = fopen(tmp, "r")) != NULL) {
      for (i=0; i<6; i++) {
         fgets(buffer, 1024, ptr);
      }     
      while (fgets(buffer, 1024, ptr) != NULL) {
         i = loc_char(buffer, 'O');  /*Look for capital 'o' */
         if (i < 0) break;

         strcpy(tmp, (char *)strstr(buffer," ORB"));
         strncpy(buffer, &tmp[4], 4);
         buffer[4] = '\0';
         sscanf(buffer, "%4d", &iorb);
         if (first) {
           options->first_orb = iorb;
           first = FALSE;
         }  
         options->last_orb = iorb;
      }  
      fclose(ptr);
      strcpy(command, "del c:\\tmp12312.asc");
      i = system(command);
   } else {
	  printf("Unable to open temporary file\n");   
   }
   return(SUCCESS);
}

/*--------------------------------------------------------------
   Determines the CD start/stop orbit numbers on non-DOS systems.
 
   Entry Requirements:           
      options:  Pointer to the options structure
 
   Return Values:                        
      SUCCESS/FAILURE
----------------------------------------------------------------*/
int
   get_orbs(
   Options *options)
{
#ifndef WIN32	/* popen()/pclose() not available on DOS/Win */
   int  i;
   int  iorb;
   int  first;
   char buffer[1024];
   char tmp[1024];
   char command[1024];
   FILE *ptr;

   first = TRUE;
   strcpy(command, "ls ");
   strcat(command, options->mag_path);
   strcat(command, "orb*.lbl");
   if ((ptr = popen(command, "r")) != NULL) {
      while (fgets(buffer, 1024, ptr) != NULL) {
         strcpy(tmp, (char *)strstr(buffer,"orb"));
         while ((i=strlen(tmp)) > 12) {
            strcpy(buffer, &tmp[4]);
            strcpy(tmp, (char *)strstr(buffer, "orb"));
         }
         strcpy(buffer, &tmp[3]);
         sscanf(buffer, "%d", &iorb);
         if (first) {
           options->first_orb = iorb;
           first = FALSE;
         }
         options->last_orb = iorb;
      }
      if ((i = pclose(ptr)) != 0) {
         if (options->file_type != FLOW) {
             msg_exit2(ERR_BAD_PATH, options->cd_location);
         } else {
             msg_exit(ERR_BAD_PATH, options->cd_location);
         }  
      }
   } else {
      if ((i = pclose(ptr)) != 0) 
      if (options->file_type != FLOW) {
         msg_exit2(ERR_OUTPUT_OPEN_FAIL, "required temporary file");
      } else {
         msg_exit(ERR_OUTPUT_OPEN_FAIL, "required temporary file");
      }  
   }
#endif
   return(SUCCESS);
}

/*----------------------------------------------------------------
   Determines the CD start/stop orbit numbers on VMS systems.
 
   Entry Requirements:
      options:  Pointer to the options structure
 
   Return Values:
      SUCCESS/FAILURE
----------------------------------------------------------------*/
int
   get_vms_orbs(
   Options *options)
{

   int  i;
   int  iorb;
   int  first;
   char buffer[1024];
   char tmp[1024];
   char command[1024];
   FILE *ptr;

   first = TRUE;
   strcpy(command, "dir/col=1/out=sys$login:tmp12312.dat ");
   strcat(command, options->mag_path);
   strcat(command, "ORB*.LBL");
   i =  system(command);
   if (i <= 0) {
      if (options->file_type != FLOW) {
          msg_exit2(ERR_BAD_PATH, options->cd_location);
      } else {
          msg_exit(ERR_BAD_PATH, options->cd_location);
      }  
   }
   if ((ptr = fopen("sys$login:tmp12312.dat", "r")) != NULL) {
      for (i=0; i<3; i++) {
         fgets(buffer, 1024, ptr); 	
      }
      while (fgets(buffer, 1024, ptr) != NULL) {
         i = strlen(buffer);
         if (i <= 1) break;
         strcpy(tmp, (char *)strstr(buffer,"ORB"));
         strncpy(buffer, &tmp[3], 4);
         buffer[4] = '\0';
         sscanf(buffer, "%4d", &iorb);
         if (first) {
           options->first_orb = iorb;
           first = FALSE;
         }
         options->last_orb = iorb;
      }   
      fclose(ptr);
      strcpy(command, "del sys$login:tmp12312.dat;*");
      i = system(command);
   } else {
      if (options->file_type != FLOW) {
          msg_exit2(ERR_BAD_PATH, options->cd_location);
      } else {
          msg_exit(ERR_BAD_PATH, options->cd_location);
      }
   }
   return(SUCCESS);
}
 
 


/*----------------------------------------------------------------
   Determines the CD start/stop orbit numbers on DOS systems.                   
   Entry Requirements:
      options:  Pointer to the options structure
 
   Return Values:
     options:   first_orb, last_orb values set.
----------------------------------------------------------------*/
int
   make_filenames(
   Options *options)
{

   char  fname[13];
   char  tmp[5];
   int   loc_dot, loc;

   strcpy(options->magfile_name, options->mag_path);
   strcpy(options->ephfile_name, options->ephem_path);
   loc_dot = loc_char(options->outfile_name, '.');

   switch (options->host) {
     case SUN:
     case  HP:
       loc = loc_char(options->outfile_name, '/');
       break;
     case VAX:
       loc = loc_char(options->outfile_name, ']');
       break;
     case  PC:
       loc = loc_char(options->outfile_name, '\\');
       break;
   } 


   if (loc_dot > 0 && loc_dot > loc) {
      strcpy(options->lblfile_name, options->outfile_name);
      options->lblfile_name[loc_dot] = '\0';
      strcpy(options->hdrfile_name, options->outfile_name);
      options->hdrfile_name[loc_dot] = '\0';
      strcat(options->lblfile_name, ".lbl");
      if (options->file_type == FLAT) {
         options->outfile_name[loc_dot] = '\0';
         strcat(options->outfile_name, ".ffd ");
         options->hdrfile_name[loc_dot] = '\0';
         strcat(options->hdrfile_name, ".ffh");
      }
   } else {
      strcpy(options->lblfile_name, options->outfile_name);
      strcpy(options->hdrfile_name, options->outfile_name);
      strcat(options->lblfile_name, ".lbl"); 
      if (options->file_type == FLAT) {
         strcat(options->outfile_name, ".ffd");
         strcat(options->hdrfile_name, ".ffh");
      }
   }
   strcpy(options->product_id, &options->outfile_name[loc + 1]);
      
   sprintf(tmp, "%04d", options->orbit);
   strcpy(fname, "orb");
   strcat(fname, tmp);
   strcat(options->ephfile_name, fname);
   strcat(options->magfile_name, fname);
   strcat(options->ephfile_name, ".ffd");
   strcat(options->magfile_name, ".ffd");
   return(SUCCESS);
}
 
/*----------------------------------------------------------------
   Opens the flatfiles (mag, ephem) on the CDROM.
 
   Entry Requirements:
      options:  Pointer to the options structure
 
   Return Values:
     options:   magfptr, ephfptr
----------------------------------------------------------------*/
int
   check_orbs(
   Options *options)
{
   char msg[80];

   if (options->orbit < options->first_orb ||
       options->orbit > options->last_orb) {
       sprintf(msg, "Orbit %04d not on this CD. This CD contains orbits %04d - %04d",
          options->orbit, options->first_orb, options->last_orb); 
       if (options->file_type == FLOW) {
          msg_exit(ERR_NO_ORBIT, msg); 
       } else {
          msg_exit2(ERR_NO_ORBIT, msg); 
       } 
   }
   return(SUCCESS);
}

/*----------------------------------------------------------------
   Opens the flatfiles (mag, ephem) on the CDROM.

   Entry Requirements:
      options:  Pointer to the options structure

   Return Values:
     options:   magfptr, ephfptr
----------------------------------------------------------------*/
int
   open_datafiles(
   Options *options)
{

  switch (options->host) {
     case VAX:
       vms_data_path(options);
       get_vms_orbs(options);
       check_orbs(options);
       make_filenames(options);
       if((options->magfptr = fopen(options->magfile_name, "r")) == NULL) {
          if (options->file_type == FLOW) {
             msg_exit(ERR_INPUT_OPEN_FAIL, options->magfile_name);
          } else {
             msg_exit2(ERR_INPUT_OPEN_FAIL, options->magfile_name);
          }
       }
       if((options->ephfptr = fopen(options->ephfile_name, "r")) == NULL) {
          if (options->file_type == FLOW) {
             msg_exit(ERR_INPUT_OPEN_FAIL, options->ephfile_name);
          } else {
             msg_exit2(ERR_INPUT_OPEN_FAIL, options->ephfile_name);
          }
       }
       break;
     case SUN:
     case  HP:
       posix_data_path(options);
       get_orbs(options);
       check_orbs(options); 
       make_filenames(options);
       if((options->magfptr = fopen(options->magfile_name, "r")) == NULL) {
          if (options->file_type == FLOW) {
             msg_exit(ERR_INPUT_OPEN_FAIL, options->magfile_name);
          } else {
             msg_exit2(ERR_INPUT_OPEN_FAIL, options->magfile_name);
          }
       }
       if((options->ephfptr = fopen(options->ephfile_name, "r")) == NULL) {
          if (options->file_type == FLOW) {
             msg_exit(ERR_INPUT_OPEN_FAIL, options->ephfile_name);
          } else {
             msg_exit2(ERR_INPUT_OPEN_FAIL, options->ephfile_name);
          }
       }
       break;
     case  PC:
       msdos_data_path(options);
       get_dos_orbs(options);
       check_orbs(options);
       make_filenames(options);
       if((options->magfptr = fopen(options->magfile_name, "rb")) == NULL) {
          msg_exit2(ERR_INPUT_OPEN_FAIL, options->magfile_name);
       }
       if((options->ephfptr = fopen(options->ephfile_name, "rb")) == NULL) {
          msg_exit2(ERR_INPUT_OPEN_FAIL, options->ephfile_name);
       }
       break;
   }
  return(SUCCESS);
}
/*----------------------------------------------------------------
   Computes the SC->VSO rotation matrix.
 
   Entry Requirements:
      ephem:  Pointer to the structure containing the ephemeris data
      rot:    3x3 rotation matrix created by this function

   Return Values
      none
 
----------------------------------------------------------------*/
void rot_matrix(
   Ephem *ephem,
   double *rot)
{
   double  g;

   g = sqrt(ephem->spin_y * ephem->spin_y + ephem->spin_z * ephem->spin_z);
   rot[0] = g;
   rot[1] = 0.;
   rot[2] = -1. * ephem->spin_x;
   rot[3] = -1. * ephem->spin_x * ephem->spin_y/g;
   rot[4] = -1. * ephem->spin_z / g;
   rot[5] = -1. * ephem->spin_y;
   rot[6] = -1. * ephem->spin_x * ephem->spin_z / g;
   rot[7] =       ephem->spin_y / g;
   rot[8] = -1. * ephem->spin_z;
}
/*---------------------------------------------------------------- 
   Rotates the MAG vectors from S/C to VSO coordinates.
  
   Entry Requirements: 
      mag:    Pointer to the structure containing the magnetometer data   
      rot:    3x3 rotation matrix created by this function 
 
   Return Values 
      none
  
----------------------------------------------------------------*/ 

void rot_mag(
    Mag *mag,
    double *rot)
{
    double Bx_sc, By_sc, Bz_sc;

    Bx_sc = mag->Bx;
    By_sc = mag->By;
    Bz_sc = mag->Bz;
    mag->Bx = rot[0]*Bx_sc + rot[1]*By_sc + rot[2]*Bz_sc;
    mag->By = rot[3]*Bx_sc + rot[4]*By_sc + rot[5]*Bz_sc;
    mag->Bz = rot[6]*Bx_sc + rot[7]*By_sc + rot[8]*Bz_sc;
}

/*--------------------------------------------------------------
   Main program starts here
----------------------------------------------------------------*/
int 
main (int argc, char *argv[])
{
  int itmp;
  int stat;
  int optnum;
  int more;
  int normal = TRUE;
  char value[1024];

  double  etime, ntime;
  double  rot[9];

  int (*data_writer)() = ascii_data_writer;
  int (*lbl_writer)() = lbl_ascii_data;

  Options   options;
  Mag       mag;
  Ephem     ephem,next_ephem;
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/

/* Initialize variables */

  strcpy(options.cd_location, "");
  strcpy(options.magfile_name, "");
  strcpy(options.ephfile_name, "");
  strcpy(options.outfile_name, "");
  strcpy(options.hdrfile_name, "");
  strcpy(options.lblfile_name, "");
  options.file_type = ASCII;

/* Check proper usage  */


  if (argc < 3) {
      fprintf(stderr,"Improper usage of PVROT\n");
      fprintf(stderr,"\n");
      fprintf(stderr,"Usage: \n");
      fprintf(stderr,"  pvrot orbit=i host=string out=string file=string cd_loc=string\n");
      fprintf(stderr,"\n");
      fprintf(stderr,"Example: pvrot orb=2316 cd_loc=/cdrom/PV01_0041 host=sun out=mag2316.asc file=ascii   \n");
      fprintf(stderr,"\n");
      fprintf(stderr,"Options are separated by a single space\n");
      fprintf(stderr,"Required keywords:\n");
      fprintf(stderr,"   ORBIT: orbit number to rotate to VSO coordiantes\n");
      fprintf(stderr,"   HOST:  platform being used, valid: SUN, HP, VAX, PC\n");
      fprintf(stderr,"   OUT: name of output file (this option is case sensitive).\n");
      fprintf(stderr,"   CD_LOC: Path to the CDROM top directory\n");
      fprintf(stderr,"          (i.e /cdrom (Unix)i, d:\\ (DOS), dua0:[000000] (VMS) \n");
      fprintf(stderr,"\n");
      fprintf(stderr,"Optional Keywords:\n");
      fprintf(stderr,"   FILE_TYPE: type of output, valid: ASCII, FLATFILE, FLOW \n");
      fprintf(stderr,"       ASCII (default) generates a PDS labeled ASCII data file.\n");
      fprintf(stderr,"       FLATFILE generates a PDS labeled IGPP (v1.0) flatfile\n");

  }

/* Parse command line options */

  while ((optnum = getcmdlineopt(Cl_options, value, TRUE)) >= 0)
  {
      switch(optnum)
      {
         case 0: break; /* allow recovery for bad keyword */
         case CD_LOCATION:
	    strcpy(options.cd_location, value);
	    break;
         case ORBIT:
	    options.orbit = atoi(value);
	    break;
         case OUTFILE:
	    strcpy(options.outfile_name, value);
	    break;
         case HOST:
            scaseup(value);
            itmp = kw_match(value, Host_types);
            if(itmp >= 0) {
               PickVarFormat(itmp);
               options.host = itmp; 
            } else {
              msg_exit2(ERR_INVALID_HOST, value);
            }
	    break;
         case FILETYPE:
            scaseup(value);
            itmp = kw_match(value, File_types);
            if(itmp >= 0) {
               options.file_type = itmp;
            } else {
              msg_exit2(ERR_INVALID_FILETYPE, value);
              options.file_type = ASCII;
            }
            break;
         default:
            break;
      }
   }

/* Check command line arguments */
   if (options.host != SUN && options.file_type == FLOW) {
      msg_exit2(ERR_NO_FLOW, "FLOW files are only supported on Sun platforms.");
      options.file_type = ASCII;
   }
   if (options.file_type != FLOW) {
       if(options.orbit <= 0  || options.orbit > 3601) 
          msg_exit2(ERR_NO_OPTION, "Orbit keyword not specified or invalid");
       if(strlen(options.outfile_name) == 0 ) 
          msg_exit2(ERR_NO_OPTION, "Output file not specified");
       if(options.host < 0 ) 
          msg_exit2(ERR_NO_OPTION, "Host keyword not specified");
       if(options.orbit == 0 ) 
          msg_exit2(ERR_NO_OPTION, "Orbit keyword not specified or invalid");
   } else {
       if(options.orbit <= 0  || options.orbit > 3601) 
          msg_exit(ERR_NO_OPTION, "Orbit keyword not specified or invalid");
       if(strlen(options.outfile_name) == 0 ) 
          msg_exit(ERR_NO_OPTION, "Output file not specified");
       if(options.host < 0 ) 
          msg_exit(ERR_NO_OPTION, "Host keyword not specified");
       if(options.orbit == 0 ) 
          msg_exit(ERR_NO_OPTION, "Orbit keyword not specified or invalid");
   }
   open_datafiles(&options);
   switch (options.file_type) {
      case ASCII:
         if((options.ofptr = fopen(options.outfile_name, "w")) == NULL) {
           msg_exit2(ERR_OUTPUT_OPEN_FAIL, options.outfile_name);
         }  
         if((options.lblptr = fopen(options.lblfile_name, "w")) == NULL) {
            msg_exit2(ERR_OUTPUT_OPEN_FAIL, options.lblfile_name);
          }  
         data_writer = ascii_data_writer;
         lbl_writer = lbl_ascii_data;
         break;
      case FLAT:
         if((options.ofptr = fopen(options.outfile_name, "w"))== NULL) {
            msg_exit2(ERR_OUTPUT_OPEN_FAIL, options.outfile_name); 
         }
         if((options.hfptr = fopen(options.hdrfile_name, "w"))== NULL) {
            msg_exit2(ERR_OUTPUT_OPEN_FAIL, options.hdrfile_name); 
         }
         if((options.lblptr = fopen(options.lblfile_name, "w")) == NULL) {
            msg_exit2(ERR_OUTPUT_OPEN_FAIL, options.lblfile_name);
         }  
         data_writer = ff_data_writer;
         lbl_writer = lbl_ff_data;
         break;
      case FLOW:
#ifndef WIN32
         initpipe();
         makedesc("Sample Time (UT)","time","seconds",'T',8,0);
         makedesc("Bx - VSO",        "Bx",  "nT",     'R',4,0);
         makedesc("By - VSO",        "By",  "nT",     'R',4,0);
         makedesc("Bz - VSO",        "Bz",  "nT",     'R',4,0);
         makedesc("|B|"     ,        "Bt",  "nT",     'R',4,0);
         wdesc();
         descend();
         data_writer = flow_data_writer;
#endif
         break;
   }


/* Read ephemeris records until the first mag record is sandwiched
   in time between two ephemeris records */

/* Set output binary form to Host Type  */

   PickVarFormat(options.host); 
   stat = fread((char *)&ephem, SIZE_OF_EPHEM, 1, options.ephfptr);
   convert_ephem(&options, &ephem);
   stat = fread((char *)&mag, SIZE_OF_MAG	,1,options.magfptr);
   convert_mag(&options, &mag);
   pdstime(mag.time, options.start_time);

   more = TRUE;
   etime = mag.time - ephem.time;
   ntime = 1.;
   while ((ntime > 0.) && more){
      stat = fread((char *)&next_ephem, SIZE_OF_EPHEM, 1, options.ephfptr);
      if (stat != 1) more = FALSE;
      convert_ephem(&options, &next_ephem);
      ntime = mag.time - next_ephem.time;
	  if (ntime > 0.) {
		  memcpy((char *)&ephem, (char *)&next_ephem, SIZE_OF_EPHEM);
		  etime = mag.time - ephem.time;
      }
   }
   if (fabs(etime) < fabs(ntime)) {
      rot_matrix(&ephem, rot);
   } else {
      rot_matrix(&next_ephem, rot);
      memcpy((char *)&ephem, (char *)&next_ephem, SIZE_OF_EPHEM);
      stat = fread((char *)&next_ephem, SIZE_OF_EPHEM, 1, options.ephfptr);
	  if (stat != 1) more = FALSE;
      convert_ephem(&options, &next_ephem);
   }

   options.num_recs = 0;
   rot_mag(&mag, rot);
   write_data(&options, &mag, data_writer);
   options.num_recs++;

   for(;;) {
	   stat=fread((char *)&mag, SIZE_OF_MAG,1,options.magfptr);
	   if (stat != 1) break;
       convert_mag(&options, &mag);
       etime = mag.time - ephem.time;
       ntime = mag.time - next_ephem.time;
       if (fabs(ntime) < fabs(etime)) {
          rot_matrix(&next_ephem, rot);
          memcpy((char *)&ephem, (char *)&next_ephem, SIZE_OF_EPHEM);
          stat = fread((char *)&next_ephem,SIZE_OF_EPHEM,1,options.ephfptr);
		  if (stat != 1) {
              more = FALSE;
          }
          convert_ephem(&options, &next_ephem);
       }
       rot_mag(&mag, rot);
       write_data(&options, &mag, data_writer);
       options.num_recs++;
   }


   pdstime(mag.time, options.stop_time);
   if (options.file_type == FLOW) {
#ifndef W32
      dataend();
#endif
   } else {
      if (options.file_type == FLAT) write_ff_hdr(&options);
      write_label(&options, lbl_writer);
   }

/* Check for read errors during processing */

   normal = TRUE;
   if (!(feof(options.magfptr)) && options.file_type == FLOW) {
     msg_exit(ERR_FILE_READ, 
          "Read error prior to End-of-File in Mag data");
     normal = FALSE;
   } else {
     if (!(feof(options.magfptr)) && options.file_type != FLOW) {
        fprintf(stderr, 
          "Read error before End-of-File reached on Mag data file\n");
        normal = FALSE;
     } else {
          fprintf(stderr,"End-of-File reached on Mag data file\n");
     }
   }

   if (ferror(options.ephfptr) && options.file_type == FLOW) {
     msg_exit(ERR_FILE_READ, 
          "Read error prior to End-of-File in Ephemeris data");
     normal = FALSE;
   } else {
     if (ferror(options.ephfptr) && options.file_type != FLOW) {
        fprintf(stderr, 
         "Read error before EOF reached on Ephemeris data file\n");
        normal = FALSE;
     } else {
        if (options.file_type != FLOW)
          fprintf(stderr,"No Ephemeris data file read errors detected\n");
     }
   }
   fprintf(stderr,"\n");
   fprintf(stderr,"\n");
   if (normal) fprintf(stderr,"Normal program termination!!\n");
   fclose (options.magfptr);
   fclose (options.ephfptr);
   fclose (options.ofptr);
   fclose (options.lblptr);
#ifndef VMS
   if(options.file_type == FLOW) {
     return(0);
   } else {
     exit(0);
   }
#endif
}
