/*---------------------------------------------------------------*/
/*   Copyright (c) 1989 Regents of the University of California  */
/*                 All Rights Reserved                           */
/*---------------------------------------------------------------*/
/*
      %W%  (UCLA, IGPP) %G%
*/
/*-- ffadd_desc --------------------------------------------
  Adds one or more fields to the flatfile structure.
  The field definitions are passed as an array of type DESC 
  which has a sentinal element with a zero length longname marking 
  the end of the list.

  NOTE: Character fields are increased in size if they are
        are defined to have widths which are multiple of 
        an int type. This is required by some systems, so we
        do it on all systems.

  Entry Requirements:
    ffptr: A flat file pointer (as returned from ffopen() )
    desc: A array of decriptor structures filled with the descriptive
	  information for each field. The list is ended with
	  a descriptor structure with a zero length longname.

  Return Value:
    FF_SUCCESS: If the descriptor was added.
    FF_FAILURE: If the ffptr is invalid or unable to alloc
    memory for the descriptor.
 
  Development History:
     Begun: 11/15/89 - Todd King
------------------------------------------------------------*/
#include "ffio.h"

int 
ffadd_desc (FLATFILE *ffptr, FF_DESC desc[])
{
  int i = 0;
  int cnt;
  int org_ncols;
  int width;
  FF_FULL_DESC *desc_ptr;
  FF_FULL_DESC *tmp_ptr;
  char buffer[FF_DESCWIDTH];

  if(ffptr == NULL) return(FF_FAILURE);

  cnt = 0;
  while(strlen(desc[cnt].longname) != 0) { cnt++; }
  org_ncols = ffptr->ncols;

  /* Now that we know how many descriptors create an array to hold
     them and process the passed list .
  */
  if((desc_ptr = (FF_FULL_DESC *)malloc(sizeof(FF_FULL_DESC) * cnt)) == NULL) {
    return(FF_FAILURE);
  }

/* Now fill in the value for each descriptor.
   We have to be care about lengths so that we transfer string
   terminators (zero bytes) as well into the elements in 'desc_ptr'.
   That is why the strncpy() has +1 for number of bytes.
*/
  for(i = 0; i < cnt; i++) {
    strncpy(buffer, desc[i].longname, FF_LNAME);
    buffer[FF_LNAME] = '\0';
    trimspc(buffer);
    strncpy(desc_ptr[i].longname, buffer, FF_LNAME + 1);

    strncpy(buffer, desc[i].shortname, FF_SNAME);
    buffer[FF_SNAME] = '\0';
    trimspc(buffer);
    strncpy(desc_ptr[i].shortname, buffer, FF_SNAME + 1);

    strncpy(buffer, desc[i].units, FF_UNITS);
    buffer[FF_UNITS] = '\0';
    trimspc(buffer);
    strncpy(desc_ptr[i].units, buffer, FF_UNITS + 1);

    desc_ptr[i].type = desc[i].type;
    width = desc[i].width;
    if(desc_ptr[i].type == FF_ALPHA) {	/* Insure that field is
                                          a multiple of an int */
      if( (width % sizeof(int)) != 0) { /* pad it */
         width += sizeof(int) - (width % sizeof(int));
      }
    }
    desc_ptr[i].width = width;
    desc_ptr[i].sort = desc[i].sort;

    desc_ptr[i].offset = ffptr->reclen;
    ffptr->reclen += desc_ptr[i].width;
    if(i != 0) desc_ptr[i - 1].next = &desc_ptr[i];
  }
  ffptr->ncols += cnt;
  desc_ptr[cnt - 1].next = NULL;    /* End link list */

  /* Now merge this into the existing descriptor */
  if(org_ncols == 0) {
    ffptr->descdb = desc_ptr;
  } else {
    tmp_ptr = ffptr->descdb;
    for(i = 1; i < org_ncols; i++) {	/* Scan to last entry */
      tmp_ptr = tmp_ptr->next;
    }
    tmp_ptr->next = &desc_ptr[0];
  }

  /* Now create a record buffer which will hold one record */
  if(ffptr->rec_buff != NULL) { free(ffptr->rec_buff); }
  if((ffptr->rec_buff = (char *)malloc(ffptr->reclen)) == NULL) {
    return(FF_FAILURE);
  }

  return(FF_SUCCESS);
}
