/*---------------------------------------------------------------*/
/*   Copyright (c) 1992 Regents of the University of California  */
/*                 All Rights Reserved                           */
/*---------------------------------------------------------------*/
/*
      %W%  (UCLA, IGPP) %G%
*/
/*-- ffupdate ------------------------------------------
  Performs all deletions and insertions indicated by calls
  to ffdelete_rec() and ffinsert_rec().

  Entry Requirements:
     ffptr: A pointer to a flatfile (as returned from ffopen()).

  Return Value:
      FF_SUCCESS if the operation was successful. FF_FAILURE if
      the flatfile is not opened for update or if ffptr is NULL.

  Development History:
     Begun: 1/14/92 - Todd King
-------------------------------------------------------------*/
#include <sys/types.h>
#include <sys/stat.h>	/* for stat() */
#include <string.h>	/* for strrchr() */
#include "ffio.h"

static int ulong_compare();
static int ffrecord_compare();

int 
ffupdate (FLATFILE *ffptr)
{
   unsigned long i;
   char buffer[MAX_FILENAME + 1];
   char *tmpffname;
   char oldffname[MAX_FILENAME + 1];
   char *ptr;
   int more;
   struct stat stat;
   FILE *fptr;

   if(ffptr == NULL) return(FF_FAILURE);

   switch(ffptr->open_style) {
      case FF_OPEN_APPEND:
      case FF_OPEN_WRITE:
	 return(ffwrite_hed(ffptr));
      case FF_OPEN_READ:
	 return(FF_SUCCESS);	/* Shouldn't change anything */
   }

/* Everything after here applies on when the opened for update */
   if(ffptr->open_style != FF_OPEN_UPDATE) return(FF_FAILURE);

   if(ffptr->dlist.cnt > 0 || ffptr->ilist.cnt > 0) {	/* Something to do */

      if(ffptr->dlist.cnt > 0) {
	 qsort(ffptr->dlist.recnum, ffptr->dlist.cnt, 
	       sizeof(unsigned long), ulong_compare);	/* Sort delete list */
      } else {	/* Must be something to insert */
	 qsort(ffptr->ilist.record, ffptr->ilist.cnt, 
	       sizeof(FF_RECORD), ffrecord_compare);	/* Sort insert list */
      }

      strcpy(buffer, ffptr->fname);
      ptr = strrchr(buffer, '/');
      if(ptr != NULL) *ptr = '\0';
      else strcpy(buffer, ".");

      tmpffname = tempnam(buffer, "FF");
      fptr = fopen(tmpffname, "w+b");

      i = 0;
      more = TRUE;
      ffmove_to_rec(ffptr, 0);
      while(ffread_rec(ffptr) == FF_SUCCESS) {
	 if( more ) {	/* If there are inserts or deletes. */
            if(ffptr->dlist.cnt > 0) {	/* Something to delete */
               if(ffptr->cur_recnum - 1 == ffptr->dlist.recnum[i]) {
		  i++;
		  if(i == ffptr->dlist.cnt) { more = FALSE; }
		  continue;	/* Skip this record */
               }
            } else {	/* Something to insert */
               while( more &&
                     ffptr->cur_recnum - 1 == ffptr->ilist.record[i].recnum) {
                  if(fwrite(ffptr->ilist.record[i].data, ffptr->reclen, 
			1, fptr) == 0) {
		     fclose(fptr);
		     unlink(tmpffname);
		     return(FF_FAILURE);
		  }
		  i++;
		  if(i == ffptr->ilist.cnt) { more = FALSE; }
               }
            }
	 }
	/* Move a record from the flatfile to the tmp file */

	 if(fwrite(ffptr->rec_buff, ffptr->reclen, 1, fptr) == 0) {
	    fclose(fptr);
	    unlink(tmpffname);
	    return(FF_FAILURE);
	 }
      }
      fclose(ffptr->data);
      strcpy(oldffname, ffptr->fname);
      strcat(oldffname, ".DAT");
      if(unlink(oldffname) == -1) { /* Failed  - cleanup */
	 unlink(tmpffname);
	 return(FF_FAILURE);
      }
      rename(tmpffname, oldffname);

      /* Clean up delete/insert array */

      if(ffptr->dlist.cnt > 0) {
	 free(ffptr->dlist.recnum);
	 ffptr->dlist.recnum = NULL;
	 ffptr->dlist.cnt = 0;
	 ffptr->dlist.max = 0;
      } else {
	 free(ffptr->ilist.record);
	 ffptr->ilist.record = NULL;
	 ffptr->ilist.cnt = 0;
	 ffptr->ilist.max = 0;
      }

      /* Update flatfile information */

      ffptr->data = fptr;
      fseek(ffptr->data, 0L, 0);
      fstat(fileno(ffptr->data), &stat);
      ffptr->nrecs = stat.st_size / ffptr->reclen;
      ffptr->cur_recnum = 0;
      ffwrite_hed(ffptr);
   }

   return(FF_SUCCESS);
}

/*-------------------------------------------------------------------
   Compares two unsigned long ints and returns the difference.
   Called by qsort().
---------------------------------------------------------------------*/
static int 
ulong_compare (unsigned long *i, unsigned long *j)
{
   return(*i - *j);
}

/*-------------------------------------------------------------------
   Compares two FF_RECORD and returns the difference of the
   record number (recnum) element.
   Called by qsort().
---------------------------------------------------------------------*/
static int 
ffrecord_compare (FF_RECORD *i, FF_RECORD *j)
{
   return(i->recnum - j->recnum);
}
