/*
      @(#)sedr2asc.c	1.1         (UCLA/IGPP)      7/20/94
*/
/*--------------------------------------------------------------------

------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h> 
#include <math.h>
#include "highpipec.h"
#include "ffio.h"
#include "BC.h"
#include "sedr.h"
#define  MAX_REC_LEN 5000
#define  TRUE 1
#define  FALSE 0

int scaseup();
int kw_match();


char *Cl_options[] = {"INFILE", "ASCFILE", "HOST",  "FILETYPE", "COLFILE", ""};
enum {INFILE = 1, ASCFILE, HOST, FILETYPE, COLFILE}; 

char *File_types[] = {"PULSE", "SEL_ROLL", "EPHEM", "MAG_EPHEM", "ATTITUDE", "LOGISTIC", "SPIN", "SC_TO_VSO_MAT",""};
enum {PULSE = 0, SEL_ROLL, EPHEM, MAG_EPHEM, ATTITUDE, LOGISTIC, SPIN, SC_TO_VSO_MAT};

char *Host_types[] = {"SUN", "VAX", "HP", "MSB", "PC", "IBM", ""};
enum {SUN = 0, VAX, HP, MSB, PC, IBM};

static   int         host;
static   char        clear_screen[30];

/*----------------------------------------------------------
   Prints a string followed by a 4 byte integer then a 
   newline to the given file pointer.
 -----------------------------------------------------------*/
void
fprint4(
   FILE *fptr,
   char	*string,
   int4	i     
) {
   fprintf(fptr, "%s", string);
#if defined(_MSDOS)
   fprintf(fptr, "%ld\n", i);
#else
   fprintf(fptr, "%d\n", i);
#endif
}

/*-------------------------------------------------------
  This is where we begin.
---------------------------------------------------------*/
int 
main (int argc, char *argv[])
{
  FILE *ifptr;
  FILE *ofptr;
  FILE *colptr;
  int itmp;
  int optnum;
  int colfile = FALSE;
  char infile_name[512];
  char outfile_name[512];
  char colfile_name[512];
  int file_type = 0;
  char value[1024];
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/

/* Initialize variables */
  strcpy(infile_name, "");
  strcpy(outfile_name, "");
  strcpy(colfile_name, "");

  host = -1;
/* Parse command line options */

  while ((optnum = getcmdlineopt(Cl_options, value, TRUE)) >= 0)
  {
      switch(optnum)
      {
         case 0: break; /* allow recovery for bad keyword */
         case INFILE:
	    strcpy(infile_name, value);
	    break;
         case ASCFILE:
	    strcpy(outfile_name, value);
	    break;
         case COLFILE:
	    strcpy(colfile_name, value);
	    break;
         case HOST:
            scaseup(value);
            itmp = kw_match(value, Host_types);
            if(itmp >= 0) {
               host = itmp;   
            } 
            else {
              perror("invalid host type:");
            }
	    break;
         case FILETYPE:
            scaseup(value);
            itmp = kw_match(value, File_types);
            if(itmp >= 0) {
               file_type = itmp;
            }
            break;
         default:
            break;
      }
   }

/* Check command line arguments */
   if(strlen(infile_name) == 0 || strlen(outfile_name) == 0   ||
      file_type < 0 || host < 0) {
	 usage(TRUE);
   }
/* Open data file specified on command line. */ 
  if(strcmp(infile_name, "-") == 0) {       /* Read from standard in */ 
      ifptr = stdin; 
  } else if((ifptr = fopen(infile_name, "rb")) == NULL) {
     perror(infile_name);
     exit(1);
  }

/* Open ascii file for output */
  if(strcmp(outfile_name, "-") == 0) {       /* Write to standard out */ 
      ofptr = stdout; 
  } else if((ofptr = fopen(outfile_name, "w")) == NULL) {
     perror(outfile_name);
     exit(1);
  }

  if (strlen(colfile_name) > 0) {
     if((colptr = fopen(colfile_name, "r")) == NULL) {
        perror(colfile_name);
        exit(1);
     } else colfile = TRUE;
  }

  switch(file_type)
  {
         case PULSE:
	    pulse(ifptr, ofptr);
	    break;
         case EPHEM:
	    ephem(ifptr, ofptr, colptr, colfile, outfile_name);
	    break;
         case MAG_EPHEM:
	    mag_ephem(ifptr, ofptr);
	    break;
         case ATTITUDE:
	    attitude(ifptr, ofptr);
	    break;
         case LOGISTIC:
	    logistic(ifptr, ofptr);
	    break;
         case SPIN:
	    spin(ifptr, ofptr);
	    break;
         case SEL_ROLL:
	    sel_roll(ifptr, ofptr);
	    break;
         case SC_TO_VSO_MAT:
	    sc_to_vso_mat(ifptr, ofptr);
	    break;
  }

  fclose(ifptr); 
  fclose(ofptr);   
  
  return(0);

}
/****************************************************************************

function PULSE

This function reads in a PVO SEDR IBM binary Pulse time file and writes out an
ascii version of the file.

****************************************************************************/
void
pulse(FILE *ifptr, FILE *ofptr)
{
   PULSE_HDR  pulse_hdr;
   PULSE_DATA pulse_rec;
   int        stat;
   int        i;
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/
   fread((char *)&pulse_hdr, sizeof(PULSE_HDR), 1, ifptr);
   convert_pulse_hdr(&pulse_hdr);
   write_pulse_hdr(ofptr, &pulse_hdr);
   fprintf(ofptr, "year doy  msec-of-Fs year doy msec-of-rip year doy msec-of-ram year doy msec-of-nad nearest_real_sun_pulse strm clck srrm fire\n");
   for (i = 0; i < pulse_hdr.num_recs; i++)
   {
      if (stat = fread((char *)&pulse_rec, sizeof(PULSE_DATA),1,ifptr) <= 0)
      {
         break;
      }
      convert_pulse_rec(&pulse_rec);
      write_pulse_rec(ofptr, &pulse_rec);
   }
}
/****************************************************************************
 
function CONVERT_PULSE_HDR

This function converts a binary IBM Pulse time header file and converts it
to IEEE (ascii).

****************************************************************************/
int 
convert_pulse_hdr (PULSE_HDR *pulse_hdr)
{
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/

/* Set input binary form to IBM 370  */
  
 BCIBM370Format(); 
 
 BCIntegerToIEEE(&pulse_hdr->lh_word);
 BCIntegerToIEEE(&pulse_hdr->num_recs);
 BCShortToIEEE(&pulse_hdr->scid);
 BCIntegerToIEEE(&pulse_hdr->orbit);
 BCShortToIEEE(&pulse_hdr->start_year);
 BCShortToIEEE(&pulse_hdr->start_doy);
 BCIntegerToIEEE(&pulse_hdr->stop_msec);
 BCIntegerToIEEE(&(pulse_hdr->start_msec));
 BCShortToIEEE(&pulse_hdr->stop_year);
 BCShortToIEEE(&pulse_hdr->stop_doy);

/* Set output binary form to Host Type  */

 PickVarFormat(host);

 BCIEEEToInteger(&pulse_hdr->lh_word);
 BCIEEEToInteger(&pulse_hdr->num_recs);
 BCIEEEToShort(&pulse_hdr->scid);
 BCIEEEToInteger(&pulse_hdr->orbit);
 BCIEEEToShort(&pulse_hdr->start_year);
 BCIEEEToShort(&pulse_hdr->start_doy);
 BCIEEEToInteger(&pulse_hdr->stop_msec);
 BCIEEEToInteger(&(pulse_hdr->start_msec));
 BCIEEEToShort(&pulse_hdr->stop_year);
 BCIEEEToShort(&pulse_hdr->stop_doy); 
 return(1);
}
/****************************************************************************

function WRITE_PULSE_HDR

This function writes the PULSE_HDR data to the output file.

****************************************************************************/
void
write_pulse_hdr (FILE *ofptr, PULSE_HDR *pulse_hdr)
{
short      block_length;
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/
block_length = pulse_hdr->lh_word.block_lenA<<6|pulse_hdr->lh_word.block_lenB;
fprintf(ofptr, "record length:     %d\n",block_length);
fprintf(ofptr, "block length:      %d\n",pulse_hdr->lh_word.rec_length);
fprintf(ofptr, "records per block: %d\n",pulse_hdr->lh_word.recs_per_block);
fprintf(ofptr, "file id:           %d\n",pulse_hdr->lh_word.file_id);
fprint4(ofptr, "number of records: ",    pulse_hdr->num_recs);
fprintf(ofptr, "S/C id:            %d\n",pulse_hdr->scid);
fprint4(ofptr, "orbit:             ",    pulse_hdr->orbit);
fprintf(ofptr, "start year:        %d\n",pulse_hdr->start_year);
fprintf(ofptr, "start day of year: %d\n",pulse_hdr->start_doy);
fprint4(ofptr, "start msec of day: ",    pulse_hdr->start_msec);
fprintf(ofptr, "stop year:         %d\n",pulse_hdr->stop_year);
fprintf(ofptr, "stop day of year:  %d\n",pulse_hdr->stop_doy);
fprint4(ofptr, "stop msec of day:  ",    pulse_hdr->stop_msec);
}
/****************************************************************************

function CONVERT_PULSE_DATA

This function converts a binary IBM Pulse time data record and converts it
to IEEE (ascii).

****************************************************************************/
int 
convert_pulse_rec (PULSE_DATA *pulse_data)
{
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/

/* Set input binary form to IBM 370  */
  
 BCIBM370Format(); 

 BCShortToIEEE(&pulse_data->year_of_Fs);
 BCShortToIEEE(&pulse_data->doy_of_Fs);
 BCShortToIEEE(&pulse_data->year_of_rip);
 BCShortToIEEE(&pulse_data->doy_of_rip);
 BCShortToIEEE(&pulse_data->year_of_ram);
 BCShortToIEEE(&pulse_data->doy_of_ram);
 BCShortToIEEE(&pulse_data->year_of_nadir);
 BCShortToIEEE(&pulse_data->doy_of_nadir);
 BCIntegerToIEEE(&pulse_data->msec_of_Fs);
 BCIntegerToIEEE(&pulse_data->msec_of_rip);
 BCIntegerToIEEE(&pulse_data->msec_of_ram);
 BCIntegerToIEEE(&pulse_data->msec_of_nadir);
 BCIntegerToIEEE(&pulse_data->pt_word);
 BCDoubleToIEEE(&pulse_data->msec_since_last_real_sun_pulse);

/* Set output binary form to Host Type  */

 PickVarFormat(host);

 BCIEEEToShort(&pulse_data->year_of_Fs);
 BCIEEEToShort(&pulse_data->doy_of_Fs);
 BCIEEEToShort(&pulse_data->year_of_rip);
 BCIEEEToShort(&pulse_data->doy_of_rip);
 BCIEEEToShort(&pulse_data->year_of_ram);
 BCIEEEToShort(&pulse_data->doy_of_ram);
 BCIEEEToShort(&pulse_data->year_of_nadir);
 BCIEEEToShort(&pulse_data->doy_of_nadir);
 BCIEEEToInteger(&pulse_data->msec_of_Fs);
 BCIEEEToInteger(&pulse_data->msec_of_rip);
 BCIEEEToInteger(&pulse_data->msec_of_ram);
 BCIEEEToInteger(&pulse_data->msec_of_nadir);
 BCIEEEToInteger(&pulse_data->pt_word);
 BCIEEEToDouble(&pulse_data->msec_since_last_real_sun_pulse); 
 return(1);
}
/****************************************************************************

function WRITE_PULSE_DATA

This function writes the PULSE_DATA data to the output file.

****************************************************************************/
void
write_pulse_rec (FILE *ofptr, PULSE_DATA *pulse_data)
{
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/
fprintf(ofptr,
#if defined(_MSDOS)
   "%4d %3d %11ld %4d %3d %11ld %4d %3d %11ld %4d %3d %11ld %22.5f %1ld    %1ld    %1ld    %1ld\n",
#else 
   "%4d %3d %11d %4d %3d %11d %4d %3d %11d %4d %3d %11d %22.5f %1d    %1d    %1d    %1d\n",
#endif
    pulse_data->year_of_Fs, 
    pulse_data->doy_of_Fs, 
    pulse_data->msec_of_Fs,
    pulse_data->year_of_rip, 
    pulse_data->doy_of_rip, 
    pulse_data->msec_of_rip,
    pulse_data->year_of_ram, 
    pulse_data->doy_of_ram, 
    pulse_data->msec_of_ram,
    pulse_data->year_of_nadir, 
    pulse_data->doy_of_nadir, 
    pulse_data->msec_of_nadir,
    pulse_data->msec_since_last_real_sun_pulse,
    pulse_data->pt_word.strm,
    pulse_data->pt_word.clck,
    pulse_data->pt_word.srrm,
    pulse_data->pt_word.fire);
}
/****************************************************************************

function SPIN

This function reads in a PVO SEDR IBM binary Spin Period file and writes out an
ascii version of the file.

****************************************************************************/
void
spin( FILE *ifptr, FILE *ofptr)
{
   SPIN_HDR  spin_hdr;
   SPIN_DATA spin_rec;
   int        stat , i;
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/
   fread((char *)&spin_hdr, sizeof(SPIN_HDR), 1, ifptr);
   convert_spin_hdr(&spin_hdr);
   write_spin_hdr(ofptr, &spin_hdr);
   fprintf(ofptr, "year doy msec-of-day    spin-period        time-delay\n");
   for (i=0;i<spin_hdr.num_recs;i++)
   {
      if (stat = fread((char *)&spin_rec, sizeof(SPIN_DATA),1,ifptr) <= 0)
      {
         break;
      }
      convert_spin_rec(&spin_rec);
      write_spin_rec(ofptr, &spin_rec);
   }
}
/****************************************************************************

function CONVERT_SPIN_HDR

This function converts a binary IBM Spin rate header record and converts it
to IEEE (ascii).

****************************************************************************/
int 
convert_spin_hdr (SPIN_HDR *spin_hdr)
{
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/

/* Set input binary form to IBM 370  */
  
 BCIBM370Format(); 

 BCIntegerToIEEE(&spin_hdr->lh_word);
 BCIntegerToIEEE(&spin_hdr->num_recs);
 BCShortToIEEE(&spin_hdr->scid);
 BCIntegerToIEEE(&spin_hdr->orbit);
 BCShortToIEEE(&spin_hdr->start_year);
 BCShortToIEEE(&spin_hdr->start_doy);
 BCShortToIEEE(&spin_hdr->stop_year);
 BCShortToIEEE(&spin_hdr->stop_doy);
 BCIntegerToIEEE(&spin_hdr->stop_msec);
 BCIntegerToIEEE(&spin_hdr->start_msec);

/* Set output binary form to Host Type  */

 PickVarFormat(host);

 BCIEEEToInteger(&spin_hdr->lh_word);
 BCIEEEToInteger(&spin_hdr->num_recs);
 BCIEEEToShort(&spin_hdr->scid);
 BCIEEEToInteger(&spin_hdr->orbit);
 BCIEEEToShort(&spin_hdr->start_year);
 BCIEEEToShort(&spin_hdr->start_doy);
 BCIEEEToShort(&spin_hdr->stop_year);
 BCIEEEToShort(&spin_hdr->stop_doy);
 BCIEEEToInteger(&spin_hdr->stop_msec);
 BCIEEEToInteger(&spin_hdr->start_msec); 
 return(1);
}
/****************************************************************************

function WRITE_SPIN_HDR

This function writes the SPIN_HDR data to the output file.

****************************************************************************/
void
write_spin_hdr (FILE *ofptr, SPIN_HDR *spin_hdr)
{
 short    block_length;
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/
block_length = spin_hdr->lh_word.block_lenA<<6|spin_hdr->lh_word.block_lenB;
fprintf(ofptr, "record length:     %d\n",spin_hdr->lh_word.rec_length);
fprintf(ofptr, "block length:      %d\n",block_length);
fprintf(ofptr, "records per block: %d\n",spin_hdr->lh_word.recs_per_block);
fprintf(ofptr, "file id:           %d\n",spin_hdr->lh_word.file_id);
fprint4(ofptr, "number of records: ",    spin_hdr->num_recs);
fprintf(ofptr, "S/C id:            %d\n",spin_hdr->scid);
fprint4(ofptr, "orbit:             ",    spin_hdr->orbit);
fprintf(ofptr, "start year:        %d\n",spin_hdr->start_year);
fprintf(ofptr, "start day of year: %d\n",spin_hdr->start_doy);
fprint4(ofptr, "start msec of day: ",    spin_hdr->start_msec);
fprintf(ofptr, "stop year:         %d\n",spin_hdr->stop_year);
fprintf(ofptr, "stop day of year:  %d\n",spin_hdr->stop_doy);
fprint4(ofptr, "stop msec of day:  ",    spin_hdr->stop_msec);
}
/****************************************************************************

function CONVERT_SPIN_DATA

This function converts a binary IBM Spin rate data record and converts it
to IEEE (ascii).

****************************************************************************/
int 
convert_spin_rec (SPIN_DATA *spin_data)
{
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/

/* Set input binary form to IBM 370  */
  
 BCIBM370Format(); 

 BCShortToIEEE(&spin_data->year);
 BCShortToIEEE(&spin_data->doy);
 BCIntegerToIEEE(&spin_data->msecs);
 BCFloatToIEEE(&spin_data->spin_period);
 BCFloatToIEEE(&spin_data->time_delay);

/* Set output binary form to Host Type  */

 PickVarFormat(host);

 BCIEEEToShort(&spin_data->year);
 BCIEEEToShort(&spin_data->doy);
 BCIEEEToInteger(&spin_data->msecs);
 BCIEEEToFloat(&spin_data->spin_period);
 BCIEEEToFloat(&spin_data->time_delay); 
 return(1);
}
/****************************************************************************

function WRITE_SPIN_DATA

This function writes the SPIN_DATA data to the output file.

****************************************************************************/
void
write_spin_rec (FILE *ofptr, SPIN_DATA  *spin_data)
{
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/
fprintf(ofptr, 
#if defined(_MSDOS)
   "%4d %3d %11ld %14.5f %14.5f\n",
#else
   "%4d %3d %11d %14.5f %14.5f\n",
#endif
    spin_data->year, 
    spin_data->doy, 
    spin_data->msecs, 
    spin_data->spin_period, 
    spin_data->time_delay);
}
/****************************************************************************

function EPHEM

This function reads in a PVO SEDR IBM binary ephemeris data file and writes 
out an ascii version of the file.

****************************************************************************/
ephem (FILE *ifptr, FILE *ofptr, FILE *colptr, int  colfile, char name[])
{
   EPHEM_HDR   ephem_hdr;
   EPHEM_ARRAY udata;
   int         index[140];
   int         stat , i, cnt=0;
   char        hdrfile_name[516];
   FILE        *hdrptr;
   int         get_index();

/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/


   if (host == SUN) strcpy(clear_screen, "clear");
   if (host == VAX) strcpy(clear_screen, "type/page nl:");
   if (host == MSB || host == PC) strcpy(clear_screen, "cls");

   system(clear_screen);
 
   if(strcmp(name, "-") == 0) {       /* Write to standard out */
       hdrptr = ofptr;
   } else {
      strncpy(hdrfile_name, name, strlen(name));
      strcat(hdrfile_name, "_hdr");
      if((hdrptr = fopen(hdrfile_name, "w")) == NULL) {
         perror(hdrfile_name);
        exit(1);
      }
   }

   cnt = get_index(colfile, colptr, index);

   fread((char *)&ephem_hdr, sizeof(EPHEM_HDR), 1, ifptr);
   convert_ephem_hdr(&ephem_hdr);
   write_ephem_hdr(hdrptr, &ephem_hdr, index, cnt);

   for (i = 0; i < ephem_hdr.num_recs; i++)
   {
      if (stat = fread((char *)&udata, sizeof(EPHEM_ARRAY),1,ifptr) <= 0)
      {  
         break;
      }
      convert_ephem_rec(&udata);
      write_ephem_rec(ofptr, &udata, index, cnt);
   }
   return(1);
}
/****************************************************************************

function MAG_EPHEM

This function reads in a PVO SEDR IBM binary ephemeris data file and writes 
out an ascii version of the OMAG/OEFD EPHEM flatfile.

****************************************************************************/
void
mag_ephem (FILE *ifptr, FILE *ofptr)
{
   EPHEM_HDR   ephem_hdr;
   EPHEM_ARRAY udata;
   int         stat , i;
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/
   fread((char *)&ephem_hdr, sizeof(EPHEM_HDR), 1, ifptr);
   convert_ephem_hdr(&ephem_hdr);
  printf("year doy     msec  vso-pvo-x  vso-pvo-y  vso-pvo-z altitude    solar-zen celest-lat celest-lon  earth-lon  range-sun      p-lat      p-long     spin-x     spin-y     spin-z\n");
   for (i = 0; i < ephem_hdr.num_recs; i++)
   {
      if (stat = fread((char *)&udata, sizeof(EPHEM_ARRAY),1,ifptr) <= 0)
      {  
         break;
      }
      convert_ephem_rec(&udata);
      write_mag_ephem_rec(ofptr, &udata);
   }
}
/****************************************************************************

function SC2VSO_MAT


****************************************************************************/
void
sc_to_vso_mat (FILE *ifptr, FILE *ofptr)
{
   EPHEM_HDR   ephem_hdr;
   EPHEM_ARRAY udata;
   int         stat , i;
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/
   fread((char *)&ephem_hdr, sizeof(EPHEM_HDR), 1, ifptr);
   convert_ephem_hdr(&ephem_hdr);
   printf("year doy     msec       A11        A12        A31        A21        A22        A23        A31        A32        A33\n");
   for (i = 0; i < ephem_hdr.num_recs; i++)
   {
      if (stat = fread((char *)&udata, sizeof(EPHEM_ARRAY),1,ifptr) <= 0)
      {  
         break;
      }
      convert_ephem_rec(&udata);
      write_vso_mat(ofptr, &udata);
   }
}
/****************************************************************************

function CONVERT_EPHEM_HDR

This function converts a binary IBM ephemeris data header record and converts it
to IEEE (ascii).

****************************************************************************/
int 
convert_ephem_hdr (EPHEM_HDR *ephem_hdr)
{
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/

/* Set input binary form to IBM 370  */
  
 BCIBM370Format(); 

 BCIntegerToIEEE(&ephem_hdr->lh_word);
 BCIntegerToIEEE(&ephem_hdr->num_recs);
 BCShortToIEEE(&ephem_hdr->scid);
 BCIntegerToIEEE(&ephem_hdr->orbit);
 BCShortToIEEE(&ephem_hdr->start_year);
 BCShortToIEEE(&ephem_hdr->start_doy);
 BCIntegerToIEEE(&ephem_hdr->start_msec);
 BCShortToIEEE(&ephem_hdr->stop_year);
 BCShortToIEEE(&ephem_hdr->stop_doy);
 BCIntegerToIEEE(&ephem_hdr->stop_msec);

/* Set output binary form to Host Type  */

 PickVarFormat(host);

 BCIEEEToInteger(&ephem_hdr->lh_word);
 BCIEEEToInteger(&ephem_hdr->num_recs);
 BCIEEEToShort(&ephem_hdr->scid);
 BCIEEEToInteger(&ephem_hdr->orbit);
 BCIEEEToShort(&ephem_hdr->start_year);
 BCIEEEToShort(&ephem_hdr->start_doy);
 BCIEEEToInteger(&ephem_hdr->start_msec);
 BCIEEEToShort(&ephem_hdr->stop_year);
 BCIEEEToShort(&ephem_hdr->stop_doy);
 BCIEEEToInteger(&ephem_hdr->stop_msec); 
 return(1);
}
/****************************************************************************

function WRITE_EPHEM_HDR

This function writes the EPHEM_HDR data to the output file.

****************************************************************************/
void
write_ephem_hdr(FILE *ofptr, EPHEM_HDR *ephem_hdr, int index[], int cnt)
{
   int       i;
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/
    fprintf(ofptr, "  1) |S/C Event Time (UTC) - Year\n");
    fprintf(ofptr, "  2) |S/C Event Time (UTC) - Day of Year\n");
    fprintf(ofptr, "  3) |S/C Event Time (UTC) - Millisecond of Day\n");

    for (i=0; i<cnt; i++) {
       fprintf(ofptr, "%3d) %s\n",i+4, desc_lines[index[i]]);
    }
}
/****************************************************************************
function CONVERT_EPHEM_DATA

This function converts a binary IBM ephemeris data record and converts it
to IEEE (ascii).

****************************************************************************/
int 
convert_ephem_rec (EPHEM_ARRAY *ephem_data)
{
int        i;
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/

/* Set input binary form to IBM 370  */
  
 BCIBM370Format(); 
	
 BCShortToIEEE(&ephem_data->t1);
 BCShortToIEEE(&ephem_data->t2);
 BCIntegerToIEEE(&ephem_data->t3);
 for (i = 0; i < 134; i++) BCDoubleToIEEE(&ephem_data->e_array[i]);

/* Set output binary form to Host Type  */
 PickVarFormat(host);

 BCIEEEToShort(&ephem_data->t1);
 BCIEEEToShort(&ephem_data->t2);
 BCIEEEToInteger(&ephem_data->t3);
 for (i = 0; i < 134; i++) BCIEEEToDouble(&ephem_data->e_array[i]);
 return(1);
}
/****************************************************************************

function WRITE_MAG_EPHEM_REC

This function writes an ascii version of the OMAG/OEFD EPHEM flatfile
from the raw SEDR data.

****************************************************************************/
void
write_mag_ephem_rec (FILE *ofptr, EPHEM_ARRAY  *ephem_data)
{
int          i;
double       x[3], y[3], z[3], xmag ,ymag, zmag;
double       rotmat[3][3];
double       pvo_vso[3], pvo_spin[3];
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/

fprintf(ofptr,
#if defined(_MSDOS)
   "%4d %3d %8ld",
#else
   "%4d %3d %8d",
#endif
        ephem_data->t1,
        ephem_data->t2,
        ephem_data->t3);
/*
 Load the Heliocentric Venus -> Sun direction into X
  and the Heliocentric Venus velocity vector into Y
*/
for (i = 0; i < 3; i++) {
  x[i] = ephem_data->e_array[i+88];
  y[i] = ephem_data->e_array[i+91];
}
/*
  Create a Z vector where Z = X x Y
     and a Y vector where Y = Z x X

   This gives us X pointing from Venus to the Sun,
   Z is the upward normal to the Venus orbital plane,
   and Y completes the right-handed set.
*/
z[0] = x[1] * y[2] - x[2] * y[1];
z[1] = x[2] * y[0] - x[0] * y[2];
z[2] = x[0] * y[1] - x[1] * y[0];
y[0] = z[1] * x[2] - z[2] * x[1];
y[1] = z[2] * x[0] - z[0] * x[2];
y[2] = z[0] * x[1] - z[1] * x[0];
/*
  Make unit vectors of X, Y, and Z
  and load X,Y,Z as rows of the rotation matrix rotmat
  which rotates from Heliocentric to VSO coordinates.
*/
xmag = sqrt(x[0]*x[0] + x[1]*x[1] + x[2]*x[2]);
ymag = sqrt(y[0]*y[0] + y[1]*y[1] + y[2]*y[2]);
zmag = sqrt(z[0]*z[0] + z[1]*z[1] + z[2]*z[2]);
for (i=0; i<3; i++) {
   rotmat[0][i] = x[i]/xmag;
   rotmat[1][i] = y[i]/ymag;
   rotmat[2][i] = z[i]/zmag;
}
/*
   Rotate the PVO Spin Axis (59-61)
      and the PVO Position Vector (28-30)
      into VSO coordinates.
*/
for (i = 0; i < 3; i++) {
   pvo_spin[i] = rotmat[i][0] * ephem_data->e_array[59] + 
                 rotmat[i][1] * ephem_data->e_array[60] +
                 rotmat[i][2] * ephem_data->e_array[61];
   pvo_vso[i]  = (rotmat[i][0] * ephem_data->e_array[28] + 
                 rotmat[i][1] * ephem_data->e_array[29] +
                 rotmat[i][2] * ephem_data->e_array[30])/ 6050.;
}
fprintf(ofptr, " %10.6f %10.6f %10.6f %10.6f %10.6f %10.6f %10.6f %10.6f %10.6f %10.6f %10.6f %10.6f %10.6f %10.6f\n", 
pvo_vso[0], pvo_vso[1], pvo_vso[2], 
(ephem_data->e_array[34] - 6050.),
ephem_data->e_array[112],
ephem_data->e_array[7],
ephem_data->e_array[8],
ephem_data->e_array[10],
(ephem_data->e_array[4]/149674000.),
ephem_data->e_array[41],
ephem_data->e_array[42],
pvo_spin[0], pvo_spin[1], pvo_spin[2]);
}


/****************************************************************************

function WRITE_VSO_MAT

This function writes an ascii version of the OMAG/OEFD EPHEM flatfile
from the raw SEDR data.

****************************************************************************/
void
write_vso_mat (FILE *ofptr, EPHEM_ARRAY *ephem_data)
{
int          i;
double       x[3], y[3], z[3], xmag ,ymag, zmag, g;
double       rotmat[3][3];
double       vso_mat[3][3];
double       pvo_spin[3];
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/

fprintf(ofptr,
#if defined(_MSDOS)
   "%4d %3d %8ld",
#else
   "%4d %3d %8d",
#endif
        ephem_data->t1,
        ephem_data->t2,
        ephem_data->t3);
/*
 Load the Heliocentric Venus -> Sun direction into X
  and the Heliocentric Venus velocity vector into Y
*/
for (i = 0; i < 3; i++) {
  x[i] = ephem_data->e_array[i+88];
  y[i] = ephem_data->e_array[i+91];
}
/*
  Create a Z vector where Z = X x Y
     and a Y vector where Y = Z x X

   This gives us X pointing from Venus to the Sun,
   Z is the upward normal to the Venus orbital plane,
   and Y completes the right-handed set.
*/
z[0] = x[1] * y[2] - x[2] * y[1];
z[1] = x[2] * y[0] - x[0] * y[2];
z[2] = x[0] * y[1] - x[1] * y[0];
y[0] = z[1] * x[2] - z[2] * x[1];
y[1] = z[2] * x[0] - z[0] * x[2];
y[2] = z[0] * x[1] - z[1] * x[0];
/*
  Make unit vectors of X, Y, and Z
  and load X,Y,Z as rows of the rotation matrix rotmat
  which rotates from Heliocentric to VSO coordinates.
*/
xmag = sqrt(x[0]*x[0] + x[1]*x[1] + x[2]*x[2]);
ymag = sqrt(y[0]*y[0] + y[1]*y[1] + y[2]*y[2]);
zmag = sqrt(z[0]*z[0] + z[1]*z[1] + z[2]*z[2]);
for (i=0; i<3; i++) {
   vso_mat[0][i] = x[i]/xmag;
   vso_mat[1][i] = y[i]/ymag;
   vso_mat[2][i] = z[i]/zmag;
}
/*
   Rotate the PVO Spin Axis (59-61)
      and the PVO Position Vector (28-30)
      into VSO coordinates.
*/
for (i = 0; i < 3; i++) {
   pvo_spin[i] = vso_mat[i][0] * ephem_data->e_array[59] +
                 vso_mat[i][1] * ephem_data->e_array[60] +
                 vso_mat[i][2] * ephem_data->e_array[61];
}
/*   
   Use spin axis in VSO coordinates to generate the NRSC to VSO
   transformation matrix in the same way that PVROT generates
   this matrix.
*/
g = sqrt(pvo_spin[1] * pvo_spin[1] + pvo_spin[2] * pvo_spin[2]);
rotmat[0][0]  = g;
rotmat[0][1]  = 0;
rotmat[0][2]  = -1.0 * pvo_spin[0];
rotmat[1][0]  = -1.0 * pvo_spin[0] * pvo_spin[1]/ g;
rotmat[1][1]  = -1.0 * pvo_spin[2]/g;
rotmat[1][2]  = -1.0 * pvo_spin[1];
rotmat[2][0]  = -1.0 * pvo_spin[0] * pvo_spin[2]/ g;
rotmat[2][1]  =        pvo_spin[1]/g;
rotmat[2][2]  = -1.0 * pvo_spin[2];

fprintf(ofptr, 
        " %10.7f %10.7f %10.7f %10.7f %10.7f %10.7f %10.7f %10.7f %10.7f\n",
        rotmat[0][0],
        rotmat[0][1],
        rotmat[0][2],
        rotmat[1][0],
        rotmat[1][1],
        rotmat[1][2],
        rotmat[2][0],
        rotmat[2][1],
        rotmat[2][2]); 
}
/****************************************************************************
/****************************************************************************

function WRITE_EPHEM_DATA

This function writes the EPHEM_DATA data to the output file.

****************************************************************************/
write_ephem_rec (FILE *ofptr, EPHEM_ARRAY *ephem_data, int index[], int cnt)
{
int          i;
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/

fprintf(ofptr,
#if defined(_MSDOS)
   "%4d %3d %8ld",
#else
   "%4d %3d %8d",
#endif
        ephem_data->t1,
        ephem_data->t2,
        ephem_data->t3);

for(i=0; i<cnt; i++) fprintf(ofptr, "%19.7e",ephem_data->e_array[index[i]-4]);
fprintf(ofptr,"\n");
/*
for(i = 0;  i < 66; i++) fprintf(ofptr," %16.9e",ephem_data->e_array[i]);
for(i = 74; i < 133; i++) fprintf(ofptr," %19.6e",ephem_data->e_array[i]);
fprintf(ofptr," %16.9e\n",ephem_data->e_array[133]);
*/
   return(1);
}
/****************************************************************************

/****************************************************************************

function ATTITUDE

This function reads in a PVO SEDR IBM binary Attitude data file and writes 
out an ascii version of the file.

****************************************************************************/
void
attitude (FILE *ifptr, FILE *ofptr)
{
   ATT_HDR  att_hdr;
   ATT_DATA att_rec;
   int      i;
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/
   fread((char *)&att_hdr, sizeof(ATT_HDR), 1, ifptr);
   convert_att_hdr(&att_hdr);
   write_att_hdr(ofptr, &att_hdr);
   fprintf(ofptr,"year doy msec-of-day   cel-latitude  cel-longitude\n");
   for (i = 0;i < att_hdr.num_recs; i++)
   {
      if (fread((char *)&att_rec, sizeof(ATT_DATA),1,ifptr) != 1)
      {
         break;
      }
      convert_att_rec(&att_rec);
      write_att_rec(ofptr, &att_rec);
   }
   perror(" ");
}
/****************************************************************************

function CONVERT_ATT_HDR

This function converts a binary IBM S/C Attitude header record and converts it
to IEEE (ascii).

****************************************************************************/
int 
convert_att_hdr (ATT_HDR *att_hdr)
{
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/

/* Set input binary form to IBM 370  */
  
 BCIBM370Format(); 
	
 BCIntegerToIEEE(&att_hdr->lh_word);
 BCIntegerToIEEE(&att_hdr->num_recs);
 BCShortToIEEE(&att_hdr->scid);
 BCIntegerToIEEE(&att_hdr->orbit);

/* Set output binary form to Host Type  */

 PickVarFormat(host);

 BCIEEEToInteger(&att_hdr->lh_word);
 BCIEEEToInteger(&att_hdr->num_recs);
 BCIEEEToShort(&att_hdr->scid);
 BCIEEEToInteger(&att_hdr->orbit); 
 return(1);
}
/****************************************************************************

function WRITE_ATT_HDR

This function writes the ATT_HDR S/C attitude header data to the output file.

****************************************************************************/
void
write_att_hdr (FILE *ofptr, ATT_HDR  *att_hdr)
{
 short   block_length;
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/
block_length = att_hdr->lh_word.block_lenA<<6|att_hdr->lh_word.block_lenB;
fprintf(ofptr, "record length:     %d\n",att_hdr->lh_word.rec_length);
fprintf(ofptr, "block length:      %d\n",block_length);
fprintf(ofptr, "records per block: %d\n",att_hdr->lh_word.recs_per_block);
fprintf(ofptr, "file id:           %d\n",att_hdr->lh_word.file_id);
fprint4(ofptr, "number of records: ",    att_hdr->num_recs);
fprintf(ofptr, "S/C id:            %d\n",att_hdr->scid);
fprint4(ofptr, "orbit:             ",    att_hdr->orbit);
}
/****************************************************************************

function CONVERT_ATT_DATA

This function converts a binary IBM S/C attitude data record and converts it
to IEEE (ascii).

****************************************************************************/
int 
convert_att_rec (ATT_DATA *att_data)
{
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/

/* Set input binary form to IBM 370  */
  
 BCIBM370Format(); 

 BCShortToIEEE(&att_data->year);
 BCShortToIEEE(&att_data->doy);
 BCIntegerToIEEE(&att_data->msecs);
 BCFloatToIEEE(&att_data->lat);
 BCFloatToIEEE(&att_data->lon);

/* Set output binary form to Host Type  */

 PickVarFormat(host);

 BCIEEEToShort(&att_data->year);
 BCIEEEToShort(&att_data->doy);
 BCIEEEToInteger(&att_data->msecs);
 BCIEEEToFloat(&att_data->lat);
 BCIEEEToFloat(&att_data->lon); 
 return(1);
}
/****************************************************************************

function WRITE_ATT_DATA

This function writes the ATT_DATA S/C attitude data to the output file.

****************************************************************************/
void
write_att_rec (FILE *ofptr, ATT_DATA  *att_data)
{
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/
fprintf(ofptr,
#if defined(_MSDOS)
   "%4d %3d %11ld %14.5f %14.5f\n",
#else 
   "%4d %3d %11d %14.5f %14.5f\n",
#endif
    att_data->year, 
    att_data->doy, 
    att_data->msecs, 
    att_data->lat, 
    att_data->lon);
}
/****************************************************************************

function SEL_ROLL

This function reads in a PVO SEDR IBM binary selected roll reference file and 
writes out an ascii version of the file.

****************************************************************************/
void
sel_roll (FILE *ifptr, FILE *ofptr)
{
   ROLL_HDR  roll_hdr;
   ROLL_DATA roll_rec;
   int       stat , i;
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/
   fread((char *)&roll_hdr, sizeof(ROLL_DATA), 1, ifptr);
   convert_roll_hdr(&roll_hdr);
   write_roll_hdr(ofptr, &roll_hdr);
   fprintf(ofptr,"year doy msec-of-day cel-lat-p-star cel-lon-p-star error-in-simulated-srr\n");
   for (i = 0; i < roll_hdr.num_recs; i++)
   {
      if (stat = fread((char *)&roll_rec, sizeof(ROLL_DATA),1,ifptr) <= 0)
      {
         break;
      }
      convert_roll_rec(&roll_rec);
      write_roll_rec(ofptr, &roll_rec);
   }
}
/****************************************************************************

function CONVERT_ROLL_HDR

This function converts a binary IBM selected roll reference file header record 
and converts it to IEEE (ascii).

****************************************************************************/
int 
convert_roll_hdr (ROLL_HDR *roll_hdr)
{
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/

/* Set input binary form to IBM 370  */
  
 BCIBM370Format(); 

 BCIntegerToIEEE(&roll_hdr->lh_word);
 BCIntegerToIEEE(&roll_hdr->num_recs);
 BCShortToIEEE(&roll_hdr->scid);
 BCIntegerToIEEE(&roll_hdr->orbit);
 BCShortToIEEE(&roll_hdr->start_year);
 BCShortToIEEE(&roll_hdr->start_doy);
 BCIntegerToIEEE(&roll_hdr->start_msec);
 BCShortToIEEE(&roll_hdr->stop_year);
 BCShortToIEEE(&roll_hdr->stop_doy);
 BCIntegerToIEEE(&roll_hdr->stop_msec);

/* Set output binary form to Host Type  */

 PickVarFormat(host);

 BCIEEEToInteger(&roll_hdr->lh_word);
 BCIEEEToInteger(&roll_hdr->num_recs);
 BCIEEEToShort(&roll_hdr->scid);
 BCIEEEToInteger(&roll_hdr->orbit);
 BCIEEEToShort(&roll_hdr->start_year);
 BCIEEEToShort(&roll_hdr->start_doy);
 BCIEEEToInteger(&roll_hdr->start_msec);
 BCIEEEToShort(&roll_hdr->stop_year);
 BCIEEEToShort(&roll_hdr->stop_doy);
 BCIEEEToInteger(&roll_hdr->stop_msec); 
 return(1);
}
/****************************************************************************

function WRITE_ROLL_HDR

This function writes the ROLL_HDR data to the output file.

****************************************************************************/
void
write_roll_hdr (FILE *ofptr, ROLL_HDR  *roll_hdr)
{
 short    block_length;
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/
block_length = roll_hdr->lh_word.block_lenA<<6|roll_hdr->lh_word.block_lenB;
fprintf(ofptr, "record length:     %d\n",roll_hdr->lh_word.rec_length);
fprintf(ofptr, "block length:      %d\n",block_length);
fprintf(ofptr, "records per block: %d\n",roll_hdr->lh_word.recs_per_block);
fprintf(ofptr, "file id:           %d\n",roll_hdr->lh_word.file_id);
fprint4(ofptr, "number of records: ",    roll_hdr->num_recs);
fprintf(ofptr, "S/C id:            %d\n",roll_hdr->scid);
fprint4(ofptr, "orbit:             ",    roll_hdr->orbit);
fprintf(ofptr, "start year:        %d\n",roll_hdr->start_year);
fprintf(ofptr, "start day of year: %d\n",roll_hdr->start_doy);
fprint4(ofptr, "start msec of day: ",    roll_hdr->start_msec);
fprintf(ofptr, "stop year:         %d\n",roll_hdr->stop_year);
fprintf(ofptr, "stop day of year:  %d\n",roll_hdr->stop_doy);
fprint4(ofptr, "stop msec of day:  ",    roll_hdr->stop_msec);
}
/****************************************************************************

function CONVERT_ROLL_DATA

This function converts a binary IBM selected roll reference data record and 
converts it to IEEE (ascii).

****************************************************************************/
int 
convert_roll_rec (ROLL_DATA *roll_data)
{
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/

/* Set input binary form to IBM 370  */
  
 BCIBM370Format(); 

 BCShortToIEEE(&roll_data->year);
 BCShortToIEEE(&roll_data->doy);
 BCIntegerToIEEE(&roll_data->msecs);
 BCFloatToIEEE(&roll_data->cel_lat_of_pulse_star);
 BCFloatToIEEE(&roll_data->cel_lon_of_pulse_star);
 BCFloatToIEEE(&roll_data->error_in_simulated_srr);

/* Set output binary form to Host Type  */

 PickVarFormat(host);

 BCIEEEToShort(&roll_data->year);
 BCIEEEToShort(&roll_data->doy);
 BCIEEEToInteger(&roll_data->msecs);
 BCIEEEToFloat(&roll_data->cel_lat_of_pulse_star);
 BCIEEEToFloat(&roll_data->cel_lon_of_pulse_star);
 BCIEEEToFloat(&roll_data->error_in_simulated_srr);
 return(1);
}
/****************************************************************************

function WRITE_ROLL_DATA

This function writes the ROLL_DATA data to the output file.

****************************************************************************/
void
write_roll_rec (FILE *ofptr, ROLL_DATA  *roll_data)
{
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/
fprintf(ofptr,
#if defined(_MSDOS)
   "%4d %3d %11ld %14.5f %14.5f         %14.5f\n",
#else 
   "%4d %3d %11d %14.5f %14.5f         %14.5f\n",
#endif
    roll_data->year, 
    roll_data->doy, 
    roll_data->msecs, 
    roll_data->cel_lat_of_pulse_star, 
    roll_data->cel_lon_of_pulse_star, 
    roll_data->error_in_simulated_srr);
}
/****************************************************************************

function LOGISTIC

This function reads in a PVO SEDR IBM binary logistics file and writes out an
ascii version of the file.

****************************************************************************/
void
logistic (FILE *ifptr, FILE *ofptr)
{
   SL_HDR  sl_hdr;
   SL_DATA sl_data;
   int     stat;
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/
   fread((char *)&sl_hdr, sizeof(SL_HDR), 1, ifptr);
  
   BCIBM370Format(); 
   BCIntegerToIEEE(&sl_hdr.lh_word);
   PickVarFormat(host);
   BCIEEEToInteger(&sl_hdr.lh_word);

   write_sl_hdr(ofptr, &sl_hdr);
  
   for (;;)
   {
      if (stat = fread((char *)&sl_data, sizeof(SL_DATA),1,ifptr) <= 0)
      {
         break;
      }
      convert_sl_data(&sl_data);
      write_sl_data(ofptr, &sl_data);
   }
}
/****************************************************************************

function WRITE_SPIN_HDR

This function writes the SPIN_HDR data to the output file.

****************************************************************************/
void
write_sl_hdr (FILE *ofptr, SL_HDR *sl_hdr)
{
 short  block_length;
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/
block_length = sl_hdr->lh_word.block_lenA<<6|sl_hdr->lh_word.block_lenB;
fprintf(ofptr, "record length:     %d\n",sl_hdr->lh_word.rec_length);
fprintf(ofptr, "block length:      %d\n",block_length);
fprintf(ofptr, "records per block: %d\n",sl_hdr->lh_word.recs_per_block);
fprintf(ofptr, "file id:           %d\n",sl_hdr->lh_word.file_id);
}
/****************************************************************************

function CONVERT_SL_DATA

This function converts a binary IBM tape header record and converts it
to IEEE (ascii).

****************************************************************************/
int 
convert_sl_data (SL_DATA *sl_data)
{
  int     i;
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/
/* Set input binary form to IBM 370  */
  
 BCIBM370Format(); 

 for (i=0; i<24; i++) BCCharToASCII(&sl_data->hdr_line[i],1);

/* Set output character form to Host Type  */

 PickVarFormat(host);
 for (i=0; i<24; i++) BCASCIIToChar(&sl_data->hdr_line[i],1);
 return(1);
}
/****************************************************************************

function WRITE_SPIN_DATA

This function writes the SPIN_DATA data to the output file.

****************************************************************************/
write_sl_data (FILE *ofptr, SL_DATA *sl_data)
{
   int    i;
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/
   for (i=0; i<23; i++) fprintf(ofptr, "%c", sl_data->hdr_line[i]);
   fprintf(ofptr, "%c\n", sl_data->hdr_line[24]); 
   return(1);
}

/*-----------------------------------------------------------------------
  PickVarFormat() - selects the right conversion functions for the
     desired format.
 
  Written by  - Gilbert Hyatt
------------------------------------------------------------------------*/
int 
PickVarFormat (int host_type)
{
  switch(host_type) {
    /*  Convert to IBM 370 Format  */
    case IBM: BCIBM370Format();
              break;
    /*  Convert to VAX/VMS Format  */
    case VAX: BCVAXFormat();
              break;
    /*  Convert to HP 1000 Format  */
    case HP: BCHP1000Format();
              break;
    /*  Convert to Intel PC using Microsoft binary Format */
    case MSB: BCMSBFormat();
              break;
    /*  Convert to Intel PC using IEEE Format  */
    case PC: BCPCFormat();
              break;
    /*  Convert to Sun Format      */
    case SUN: BCSunFormat();
              break;
    /*  Non of the above           */
    default:
      return(0);
      break;
  }
  return(1);
}
 
/****************************************************************************

Function usage:

This function prints to the screen the proper usage of the program if any of
the required options are not specified.
*****************************************************************************/

int 
usage (int do_exit)
{
    printf("sedr2asc:  Converts a PVO SEDR raw IBM binary data file to an ascii data file.\n");
    printf("\n");
    printf("Usage:\n");
    printf("sedr2asc [infile=AAA] [ascfile=BBB] [host=CCC] [filetype=DDD] (colfile=EEEEE)\n");
    printf("\n");
    printf("Note:  sedr2asc [i=AAA] [a=BBB] [h=CCC] [f=DDD] (c=EEEEE) also works.\n");
    printf("\n");
    printf("Required Options:\n");
    printf("    INFILE: name of the input data file (-) for STDIN\n");
    printf("    ASCFILE: name of the output ascii data file (-) for STDOUT.\n");
    printf("    HOST: The current platform type. Valid entries are:\n");
    printf("               SUN\n");
    printf("               VAX\n");
    printf("               IBM\n");
    printf("               PC   (IBM DOS)\n");
    printf("               HP\n");
    printf("               MSB (MicroSoft DOS)\n");
    printf("    FILETYPE: type of input file. Valid file types are\n");
    printf("               PULSE\n");
    printf("               SPIN\n");
    printf("               EPHEM\n");
    printf("               MAG_EPHEM\n");
    printf("                 outputs an ascii version of the OMAG/OEFD EPHEM flatfile\n");
    printf("               ATTITUDE\n");
    printf("               SEL_ROLL\n");
    printf("               LOGISTIC\n");
    printf("               SC2VSO_MAT\n");
    printf("                 outputs a matrix which transforms from ISC to VSO coords\n");
    printf("Optional:\n");
    printf("    COLFILE: name of the ascii file containing a list of column numbers\n");
    printf("              to be extracted from the \"ephemeris\" dataset.  This file\n");
    printf("              contains 1 column number per line.\n");
    printf("\n");
    printf("All options are required except for COLFILE!!!\n");

    if(do_exit) {
      exit(1);
    }
    return(1);
}


/*******************************************************
  Function get_index gets the list of ephemeris data columns from
  a save file or from the user
*********************************************************/

int
get_index(int colfile, FILE *colptr, int index[])
{
  int     cnt;
  char   colfile_name[516];
  char   ans[20];
  int    i, j;
  int    num = 0;
  char   *value[30];
  char   *prompt = {"Enter column numbers - 'none' - or 'quit':   "};
  int	parse();
/*------------------------------------------------*/
/*        Begin Active Code                       */
/*------------------------------------------------*/
   strcpy(ans, "");
   strcpy(colfile_name, "");
   cnt = 0;
   if (colfile) {
      while (fscanf(colptr, "%d", &index[cnt]) != EOF) cnt++;
   } else {
      for (i=0; i<20; i++) printf("%s\n", explain[i]);
      scanf("%c", ans);
      i = 0;
      while ((strcmp(ans, "quit") != 0) && (i<140)) {
          system(clear_screen);
          printf("%s\n",hdr_lines[0]);
          printf("%s\n",hdr_lines[1]);
          for (j=0; j<20; j++) {
              i++;
              printf("%s\n", desc_lines[i]);
          }
          printf("\n");
          printf("%s", prompt);
          scanf("%s", ans);
          if (strcmp(ans, "quit") != 0) {
             num = parse(ans, value, ",", 20);
             for (j=0; j<num; j++) {
                index[cnt] = atoi(value[j]);
                if (index[cnt] > 0 && index[cnt] <= 137) cnt++;
             }
          }   
          if ((strcmp(ans, "quit") == 0) || (i == 140)) {
             system(clear_screen);
             printf("Selected Columns are:\n \n");
             for (j=0; j<cnt; j++) {
                 if ( (j+1)%20 == 0) {
                    printf("\n");
                    printf("Press any key and return to continue...\n");
                    scanf("%s", ans);
                    system(clear_screen);
                 }
                 else printf("%s\n",desc_lines[index[j]]);
             }
             printf("\n");
             printf("Is this correct? [y/n] ");
             scanf("%s", ans);
             if(strcmp(ans, "y") != 0) {
                cnt = 0;
                i = 0;
             } else strcpy(ans, "quit");
          }
      }    
      system(clear_screen);
      printf("Save the selected list of emphemeris parameters to a file\n");
      printf("to be re-used with the 'COLFILE' option later? [y/n]:\n");
      printf("\n");
      scanf("%s", ans);
      if(strcmp(ans, "y") == 0) {
         printf("\n");
         printf("Enter name of save file:   ");
         scanf("%s", colfile_name);
         while (!colfile) {
            if (strlen(colfile_name) > 0) {
               if((colptr = fopen(colfile_name, "w")) == NULL) {
                  printf("Error opening file: %s\n", colfile_name);
                  printf("Please enter save file name:   ");
               } else {
                 for (i=0; i<cnt; i++) fprintf(colptr," %d \n",index[i]);
                 colfile = TRUE;
               } 
            }   
         }
      }   
   }   
   return(cnt);
}
